# TaskMaster - Production-Grade CLI Task Management Tool

A comprehensive command-line task management application built with Go, demonstrating clean architecture, professional development practices, and production-ready patterns.

## Table of Contents

- [Overview](#overview)
- [Features](#features)
- [Quick Start](#quick-start)
- [Architecture Deep Dive](#architecture-deep-dive)
- [Project Structure](#project-structure)
- [Implementation Guide](#implementation-guide)
- [Source Code Walkthrough](#source-code-walkthrough)
- [Testing Strategy](#testing-strategy)
- [Deployment Guide](#deployment-guide)
- [Development Workflow](#development-workflow)
- [Advanced Usage](#advanced-usage)
- [Troubleshooting](#troubleshooting)
- [Contributing](#contributing)

## Overview

TaskMaster is a professional CLI task management tool that showcases best practices in Go development. Built with a clean 4-layer architecture, it provides a robust foundation for understanding how to structure production Go applications.

### Key Highlights

- **Clean Architecture**: Separation of concerns across Commands, Service, Repository, and Database layers
- **SQLite Persistence**: Zero-configuration database with automatic migrations
- **Rich CLI Interface**: Built with Cobra framework for professional command-line experience
- **Comprehensive Testing**: Unit and integration tests with 80%+ coverage
- **Docker Ready**: Multi-stage builds for production deployment
- **Type-Safe**: Leverages Go's type system for compile-time safety
- **Production Patterns**: Error handling, validation, and resource management

## Features

### Core Functionality

- **Task CRUD Operations**
  - Create tasks with title, description, priority, and tags
  - List tasks with powerful filtering capabilities
  - Update existing tasks (title, description, priority, status)
  - Delete tasks with soft-delete support
  - Mark tasks as complete with a single command

- **Advanced Filtering**
  - Filter by status (pending, in-progress, completed, cancelled)
  - Filter by priority level (low, medium, high, critical)
  - Filter by tags (one or multiple)
  - Full-text search across titles and descriptions
  - Combine multiple filters for precise queries

- **Rich Output**
  - Colorized terminal output for better readability
  - Tabular display with aligned columns
  - Status indicators with color coding
  - Task statistics and completion rates

- **Data Persistence**
  - SQLite database stored in `~/.taskmaster/tasks.db`
  - Automatic schema migrations
  - ACID compliance for data integrity
  - Many-to-many tag relationships

## Quick Start

### Prerequisites

- **Go 1.21+**: Download from [golang.org](https://golang.org/dl/)
- **Git** (optional): For cloning the repository
- **Docker** (optional): For containerized deployment

### Installation

#### Option 1: Build from Source

```bash
# Clone the repository
git clone https://github.com/yourusername/taskmaster.git
cd taskmaster

# Download dependencies
go mod download

# Build the binary
go build -o taskmaster cmd/taskmaster/main.go

# (Optional) Install system-wide on Unix/Linux/macOS
sudo mv taskmaster /usr/local/bin/

# Verify installation
taskmaster --help
```

#### Option 2: Using Make

```bash
# Build
make build

# Install to /usr/local/bin
make install

# Run without installing
make run
```

#### Option 3: Using Docker

```bash
# Build Docker image
docker build -t taskmaster:latest .

# Create a shell alias for convenience
alias tm='docker run -it --rm -v taskmaster-data:/root/.taskmaster taskmaster:latest'

# Use like a native command
tm add -t "My first task" -p high
tm list
```

### Basic Usage

```bash
# Add a new task
taskmaster add -t "Review pull requests" -p high --tags code-review,urgent

# List all tasks
taskmaster list

# List with filters
taskmaster list --status pending
taskmaster list --priority high
taskmaster list --tag urgent
taskmaster list --search "bug"

# Combine filters
taskmaster list --status pending --priority high --tag urgent

# Mark task as complete
taskmaster complete 1

# Update a task
taskmaster update 1 --title "New title"
taskmaster update 1 --priority 3
taskmaster update 1 --status 1

# Delete a task
taskmaster delete 1

# View statistics
taskmaster stats
```

## Architecture Deep Dive

TaskMaster follows Clean Architecture principles, organizing code into distinct layers with clear responsibilities and dependencies flowing in one direction (inward).

### Architecture Diagram

```
┌─────────────────────────────────────────────────────────────┐
│                     CLI Interface Layer                      │
│  ┌─────────────────────────────────────────────────────┐   │
│  │  Commands (Cobra)                                    │   │
│  │  - add.go, list.go, complete.go, update.go, etc.    │   │
│  │  - Flag parsing, user input validation               │   │
│  │  - Output formatting and colorization                │   │
│  └─────────────────────────────────────────────────────┘   │
└─────────────────────┬───────────────────────────────────────┘
                      │ calls
                      ▼
┌─────────────────────────────────────────────────────────────┐
│                     Service Layer                            │
│  ┌─────────────────────────────────────────────────────┐   │
│  │  Business Logic (task_service.go)                   │   │
│  │  - Input validation and sanitization                │   │
│  │  - Business rules enforcement                        │   │
│  │  - Orchestration of repository operations            │   │
│  │  - Error transformation                              │   │
│  └─────────────────────────────────────────────────────┘   │
└─────────────────────┬───────────────────────────────────────┘
                      │ calls
                      ▼
┌─────────────────────────────────────────────────────────────┐
│                   Repository Layer                           │
│  ┌─────────────────────────────────────────────────────┐   │
│  │  Data Access (task_repository.go)                   │   │
│  │  - CRUD operations                                   │   │
│  │  - Query building and optimization                   │   │
│  │  - Transaction management                            │   │
│  │  - Database-specific logic                           │   │
│  └─────────────────────────────────────────────────────┘   │
└─────────────────────┬───────────────────────────────────────┘
                      │ uses
                      ▼
┌─────────────────────────────────────────────────────────────┐
│                     Database Layer                           │
│  ┌─────────────────────────────────────────────────────┐   │
│  │  ORM & Database (GORM + SQLite)                     │   │
│  │  - Schema definition (models/task.go)               │   │
│  │  - Database connection management                    │   │
│  │  - Automatic migrations                              │   │
│  │  - Relationship handling                             │   │
│  └─────────────────────────────────────────────────────┘   │
└─────────────────────────────────────────────────────────────┘

                    Data Models (Cross-cutting)
┌─────────────────────────────────────────────────────────────┐
│  Task, Tag, Priority, Status, TaskFilter                    │
│  - Shared across all layers                                 │
│  - Type-safe enumerations                                   │
│  - Validation tags                                          │
└─────────────────────────────────────────────────────────────┘
```

### Layer Responsibilities

#### 1. Commands Layer (`internal/commands/`)

**Purpose**: Handle user interaction and CLI interface

**Responsibilities**:
- Parse command-line flags and arguments
- Validate user input at the CLI level
- Format and display output to the user
- Handle CLI-specific errors
- Provide help text and examples

**Key Characteristics**:
- Depends on Service Layer (not Repository)
- No direct database access
- Uses Cobra framework for command structure
- Implements colorized output with `fatih/color`

**Example Flow**:
```
User Input → Flag Parsing → Service Call → Format Response → Display Output
```

#### 2. Service Layer (`internal/service/`)

**Purpose**: Implement business logic and orchestration

**Responsibilities**:
- Validate business rules (e.g., non-empty titles)
- Sanitize input (trim whitespace)
- Transform data between layers
- Orchestrate multiple repository calls if needed
- Provide transaction boundaries
- Return business-friendly errors

**Key Characteristics**:
- Depends only on Repository interface
- No knowledge of CLI or database details
- Stateless (can be safely shared)
- Easy to test with mock repositories

**Example Logic**:
```go
// Business rule: Task titles must not be empty
if strings.TrimSpace(title) == "" {
    return ErrEmptyTitle
}
```

#### 3. Repository Layer (`internal/repository/`)

**Purpose**: Abstract data access and provide persistence operations

**Responsibilities**:
- CRUD operations (Create, Read, Update, Delete)
- Build complex queries with filters
- Handle N+1 query problems (eager loading)
- Manage database transactions
- Convert database errors to domain errors

**Key Characteristics**:
- Interface-based design for testability
- Uses GORM for database operations
- Optimizes queries (preloading, joins)
- Returns `ErrTaskNotFound` instead of GORM errors

**Query Optimization Example**:
```go
// Preload tags to avoid N+1 queries
query := r.db.Preload("Tags")
```

#### 4. Database Layer (GORM + SQLite)

**Purpose**: Persist data and manage database schema

**Responsibilities**:
- Store and retrieve data
- Maintain ACID properties
- Handle relationships (many-to-many)
- Automatic schema migrations
- Connection pooling

**Schema**:
```sql
-- Tasks table
CREATE TABLE tasks (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    title TEXT NOT NULL,
    description TEXT,
    priority INTEGER DEFAULT 2,
    status INTEGER DEFAULT 0,
    due_date DATETIME,
    created_at DATETIME,
    updated_at DATETIME,
    deleted_at DATETIME -- Soft delete support
);

-- Tags table
CREATE TABLE tags (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    name TEXT UNIQUE NOT NULL
);

-- Many-to-many junction table
CREATE TABLE task_tags (
    task_id INTEGER,
    tag_id INTEGER,
    FOREIGN KEY (task_id) REFERENCES tasks(id),
    FOREIGN KEY (tag_id) REFERENCES tags(id),
    PRIMARY KEY (task_id, tag_id)
);
```

### Data Flow Examples

#### Creating a Task

```
1. User: taskmaster add -t "Fix bug" -p high --tags urgent

2. Commands Layer (add.go):
   - Parse flags: title="Fix bug", priority="high", tags=["urgent"]
   - Call: taskService.CreateTask(title, description, priority, tags)

3. Service Layer (task_service.go):
   - Validate: title != ""
   - Sanitize: strings.TrimSpace(title)
   - Convert: "high" → models.PriorityHigh
   - Build: Task{Title: "Fix bug", Priority: High, Tags: [Tag{Name: "urgent"}]}
   - Call: repo.Create(task)

4. Repository Layer (task_repository.go):
   - Execute: db.Create(task)
   - GORM handles tag creation/association
   - Return: created task with ID

5. Database Layer (SQLite):
   - INSERT INTO tasks (title, priority, status, ...) VALUES (...)
   - INSERT INTO tags (name) VALUES ('urgent') ON CONFLICT DO NOTHING
   - INSERT INTO task_tags (task_id, tag_id) VALUES (...)
   - Return: auto-generated ID

6. Response flows back up:
   Repository → Service → Commands → User Output
```

#### Listing with Filters

```
1. User: taskmaster list --status pending --priority high --tag urgent

2. Commands Layer (list.go):
   - Parse flags: status="pending", priority="high", tag="urgent"
   - Build filter: TaskFilter{Status: &StatusPending, Priority: &PriorityHigh, Tag: "urgent"}
   - Call: taskService.ListTasks(filter)

3. Service Layer:
   - Pass through to repository (no business logic needed)
   - Call: repo.List(filter)

4. Repository Layer:
   - Build dynamic query:
     query := db.Preload("Tags").
              Where("status = ?", filter.Status).
              Where("priority = ?", filter.Priority).
              Joins("JOIN task_tags ON ...").
              Joins("JOIN tags ON ...").
              Where("tags.name = ?", filter.Tag).
              Order("priority DESC, created_at DESC")
   - Execute query
   - Return: []*Task

5. Database Layer:
   - Execute complex JOIN query
   - Return result set

6. Commands Layer:
   - Format as table
   - Colorize status column
   - Display to user
```

### Dependency Injection

The application uses constructor injection to wire dependencies:

```go
// main.go
func main() {
    // Initialize database
    db := initDatabase()

    // Build dependency chain
    repo := repository.NewTaskRepository(db)    // Repository depends on DB
    svc := service.NewTaskService(repo)         // Service depends on Repository
    commands.Execute(svc)                       // Commands depend on Service
}
```

**Benefits**:
- Easy to test (inject mocks)
- Clear dependency graph
- Compile-time safety
- No global state

## Project Structure

```
taskmaster/
├── cmd/
│   └── taskmaster/
│       └── main.go                 # Application entry point, dependency wiring
│
├── internal/                       # Private application code (not importable)
│   ├── commands/                   # CLI command definitions
│   │   ├── root.go                # Root command and command registration
│   │   ├── add.go                 # Add task command
│   │   ├── list.go                # List tasks command
│   │   ├── complete.go            # Complete task command
│   │   ├── update.go              # Update task command
│   │   ├── delete.go              # Delete task command
│   │   └── stats.go               # Statistics command
│   │
│   ├── service/                    # Business logic layer
│   │   └── task_service.go        # Task operations with validation
│   │
│   ├── repository/                 # Data access layer
│   │   └── task_repository.go     # Database operations and queries
│   │
│   └── models/                     # Domain models (shared across layers)
│       └── task.go                # Task, Tag, Priority, Status, TaskFilter
│
├── tests/                          # Integration tests
│   └── integration_test.go        # End-to-end test scenarios
│
├── go.mod                          # Go module definition
├── go.sum                          # Dependency checksums
├── Dockerfile                      # Multi-stage container build
├── docker-compose.yml              # Docker orchestration
├── Makefile                        # Build automation
└── README.md                       # This file
```

### File-by-File Breakdown

#### `cmd/taskmaster/main.go` (Entry Point)

**Purpose**: Bootstrap the application and wire dependencies

**Key Responsibilities**:
- Determine database path (`~/.taskmaster/tasks.db`)
- Create data directory if needed
- Initialize GORM connection
- Run database migrations
- Construct dependency chain
- Execute CLI

**Code Structure**:
```go
func main() {
    if err := run(); err != nil {
        fmt.Fprintf(os.Stderr, "Error: %v\n", err)
        os.Exit(1)
    }
}

func run() error {
    // 1. Setup paths
    // 2. Connect to database
    // 3. Run migrations
    // 4. Initialize layers
    // 5. Execute CLI
}
```

**Error Handling**: All errors bubble up to `run()`, which returns them to `main()` for consistent exit handling.

#### `internal/models/task.go` (Domain Models)

**Purpose**: Define core data structures used across all layers

**Structures**:

1. **Task**: Main entity representing a task
   ```go
   type Task struct {
       ID          uint           // Auto-incrementing primary key
       Title       string         // Task title (required)
       Description string         // Optional details
       Priority    Priority       // Priority level (0-3)
       Status      Status         // Lifecycle state (0-3)
       DueDate     *time.Time     // Optional due date
       Tags        []Tag          // Many-to-many relationship
       CreatedAt   time.Time      // Auto-populated by GORM
       UpdatedAt   time.Time      // Auto-updated by GORM
       DeletedAt   gorm.DeletedAt // Soft delete support
   }
   ```

2. **Priority**: Enumeration for priority levels
   ```go
   type Priority int
   const (
       PriorityLow      Priority = 0
       PriorityMedium   Priority = 1
       PriorityHigh     Priority = 2
       PriorityCritical Priority = 3
   )
   ```

3. **Status**: Enumeration for task states
   ```go
   type Status int
   const (
       StatusPending    Status = 0
       StatusInProgress Status = 1
       StatusCompleted  Status = 2
       StatusCancelled  Status = 3
   )
   ```

4. **Tag**: Categorization labels
   ```go
   type Tag struct {
       ID   uint
       Name string  // Unique constraint
   }
   ```

5. **TaskFilter**: Query parameters
   ```go
   type TaskFilter struct {
       Status   *Status    // nil = all statuses
       Priority *Priority  // nil = all priorities
       Tag      string     // empty = no tag filter
       Search   string     // full-text search
   }
   ```

**Design Decisions**:
- Integer enums for efficient storage and sorting
- Pointer fields for optional filters (distinguish between "not set" and "zero value")
- GORM tags for database schema generation
- JSON tags for potential API exposure
- Stringer methods for human-readable output

#### `internal/repository/task_repository.go` (Data Access)

**Purpose**: Abstract database operations behind an interface

**Interface**:
```go
type TaskRepository interface {
    Create(task *Task) error
    GetByID(id uint) (*Task, error)
    List(filter TaskFilter) ([]*Task, error)
    Update(task *Task) error
    Delete(id uint) error
    GetStats() (*Stats, error)
}
```

**Implementation Highlights**:

1. **Create**: Simple insertion with GORM
   ```go
   func (r *taskRepository) Create(task *Task) error {
       return r.db.Create(task).Error
   }
   ```
   - GORM automatically handles many-to-many tag associations
   - Tags are created or reused based on `name` uniqueness

2. **GetByID**: Single task retrieval with eager loading
   ```go
   func (r *taskRepository) GetByID(id uint) (*Task, error) {
       var task Task
       err := r.db.Preload("Tags").First(&task, id).Error
       if errors.Is(err, gorm.ErrRecordNotFound) {
           return nil, ErrTaskNotFound
       }
       return &task, err
   }
   ```
   - `Preload("Tags")` solves N+1 query problem
   - Converts GORM error to domain error

3. **List**: Dynamic query builder
   ```go
   func (r *taskRepository) List(filter TaskFilter) ([]*Task, error) {
       query := r.db.Preload("Tags")

       if filter.Status != nil {
           query = query.Where("status = ?", *filter.Status)
       }
       if filter.Priority != nil {
           query = query.Where("priority = ?", *filter.Priority)
       }
       if filter.Tag != "" {
           query = query.Joins("JOIN task_tags ON ...").
                         Joins("JOIN tags ON ...").
                         Where("tags.name = ?", filter.Tag)
       }
       if filter.Search != "" {
           pattern := "%" + filter.Search + "%"
           query = query.Where("title LIKE ? OR description LIKE ?",
                               pattern, pattern)
       }

       return query.Order("priority DESC, created_at DESC").Find(&tasks).Error
   }
   ```
   - Conditional query building based on filter
   - Efficient JOINs for tag filtering
   - LIKE queries for full-text search
   - Consistent ordering (priority first, then date)

4. **Update**: Full entity update
   ```go
   func (r *taskRepository) Update(task *Task) error {
       return r.db.Save(task).Error
   }
   ```
   - Uses `Save()` which updates all fields
   - GORM handles `updated_at` automatically

5. **Delete**: Soft delete
   ```go
   func (r *taskRepository) Delete(id uint) error {
       result := r.db.Delete(&Task{}, id)
       if result.RowsAffected == 0 {
           return ErrTaskNotFound
       }
       return result.Error
   }
   ```
   - GORM's default behavior is soft delete (sets `deleted_at`)
   - Check `RowsAffected` to detect non-existent tasks

6. **GetStats**: Aggregation queries
   ```go
   func (r *taskRepository) GetStats() (*Stats, error) {
       var stats Stats
       r.db.Model(&Task{}).Count(&stats.Total)
       r.db.Model(&Task{}).Where("status = ?", StatusCompleted).Count(&stats.Completed)
       // ... more counts
       return &stats, nil
   }
   ```
   - Multiple COUNT queries (could be optimized with GROUP BY)

**Error Strategy**:
- Return `ErrTaskNotFound` for missing records
- Let other database errors propagate (caller handles)

#### `internal/service/task_service.go` (Business Logic)

**Purpose**: Implement business rules and validation

**Interface**:
```go
type TaskService interface {
    CreateTask(title, description string, priority Priority, tags []string) (*Task, error)
    GetTask(id uint) (*Task, error)
    ListTasks(filter TaskFilter) ([]*Task, error)
    CompleteTask(id uint) error
    UpdateTask(id uint, updates map[string]interface{}) error
    DeleteTask(id uint) error
    GetStatistics() (*Stats, error)
}
```

**Implementation Highlights**:

1. **CreateTask**: Validation and transformation
   ```go
   func (s *taskService) CreateTask(title, description string, priority Priority, tags []string) (*Task, error) {
       // Validation
       title = strings.TrimSpace(title)
       if title == "" {
           return nil, ErrEmptyTitle
       }

       // Build task object
       task := &Task{
           Title:       title,
           Description: strings.TrimSpace(description),
           Priority:    priority,
           Status:      StatusPending,
       }

       // Convert tag names to Tag objects
       for _, tagName := range tags {
           tagName = strings.TrimSpace(tagName)
           if tagName != "" {
               task.Tags = append(task.Tags, Tag{Name: tagName})
           }
       }

       return s.repo.Create(task)
   }
   ```
   - Business rule: non-empty title
   - Sanitization: trim whitespace
   - Default values: status = pending
   - Tag filtering: ignore empty strings

2. **CompleteTask**: Status update shortcut
   ```go
   func (s *taskService) CompleteTask(id uint) error {
       task, err := s.repo.GetByID(id)
       if err != nil {
           return err
       }
       task.Status = StatusCompleted
       return s.repo.Update(task)
   }
   ```
   - Read-modify-write pattern
   - Could be optimized with direct UPDATE query

3. **UpdateTask**: Safe partial updates
   ```go
   func (s *taskService) UpdateTask(id uint, updates map[string]interface{}) error {
       task, err := s.repo.GetByID(id)
       if err != nil {
           return err
       }

       // Apply updates with type safety
       if title, ok := updates["title"].(string); ok && title != "" {
           task.Title = strings.TrimSpace(title)
       }
       if desc, ok := updates["description"].(string); ok {
           task.Description = strings.TrimSpace(desc)
       }
       // ... more fields

       return s.repo.Update(task)
   }
   ```
   - Type assertions for safety
   - Validation per field
   - Prevents empty title updates

**Design Patterns**:
- **Facade**: Provides simplified interface over repository
- **Template Method**: Common validation/sanitization patterns
- **Strategy**: Different update strategies per operation

#### `internal/commands/root.go` (CLI Root)

**Purpose**: Define root command and register subcommands

```go
func Execute(svc TaskService) error {
    taskService = svc  // Store service in package variable

    rootCmd := &cobra.Command{
        Use:   "taskmaster",
        Short: "A powerful CLI task management tool",
        Long:  `...`,
    }

    // Register all subcommands
    rootCmd.AddCommand(addCmd())
    rootCmd.AddCommand(listCmd())
    rootCmd.AddCommand(completeCmd())
    rootCmd.AddCommand(deleteCmd())
    rootCmd.AddCommand(updateCmd())
    rootCmd.AddCommand(statsCmd())

    return rootCmd.Execute()
}
```

**Package-Level Service**:
```go
var taskService TaskService
```
- Shared across all command files
- Set once during Execute()
- Avoids passing through every function

#### `internal/commands/add.go` (Add Command)

**Purpose**: Handle task creation from CLI

**Structure**:
```go
func addCmd() *cobra.Command {
    var (
        title       string
        description string
        priority    string
        tags        []string
    )

    cmd := &cobra.Command{
        Use:   "add",
        Short: "Add a new task",
        Long:  `...`,
        RunE: func(cmd *cobra.Command, args []string) error {
            return addTask(title, description, priority, tags)
        },
    }

    // Define flags
    cmd.Flags().StringVarP(&title, "title", "t", "", "Task title (required)")
    cmd.Flags().StringVarP(&description, "description", "d", "", "Task description")
    cmd.Flags().StringVarP(&priority, "priority", "p", "medium", "Priority level")
    cmd.Flags().StringSliceVar(&tags, "tags", []string{}, "Task tags")
    cmd.MarkFlagRequired("title")

    return cmd
}
```

**Execution Function**:
```go
func addTask(title, description, priorityStr string, tags []string) error {
    priority := parsePriority(priorityStr)

    task, err := taskService.CreateTask(title, description, priority, tags)
    if err != nil {
        return fmt.Errorf("failed to create task: %w", err)
    }

    // Colorized success output
    color.Green("✓ Task created successfully!")
    fmt.Printf("ID: %d\n", task.ID)
    fmt.Printf("Title: %s\n", task.Title)
    // ... more output

    return nil
}
```

**Helper**:
```go
func parsePriority(s string) Priority {
    switch s {
    case "low": return PriorityLow
    case "high": return PriorityHigh
    case "critical": return PriorityCritical
    default: return PriorityMedium
    }
}
```

#### `internal/commands/list.go` (List Command)

**Purpose**: Display tasks with filtering and formatting

**Flag Parsing**:
```go
cmd.Flags().StringVarP(&status, "status", "s", "", "Filter by status")
cmd.Flags().StringVarP(&priority, "priority", "p", "", "Filter by priority")
cmd.Flags().StringVar(&tag, "tag", "", "Filter by tag")
cmd.Flags().StringVar(&search, "search", "", "Search text")
```

**Filter Building**:
```go
filter := TaskFilter{
    Tag:    tag,
    Search: search,
}

// Only set status/priority if provided (pointer semantics)
if status != "" {
    s := parseStatus(status)
    filter.Status = &s
}
if priority != "" {
    p := parsePriority(priority)
    filter.Priority = &p
}
```

**Display Function**:
```go
func displayTasks(tasks []*Task) {
    if len(tasks) == 0 {
        color.Yellow("No tasks found.")
        return
    }

    // Use tabwriter for aligned columns
    w := tabwriter.NewWriter(os.Stdout, 0, 0, 2, ' ', 0)
    fmt.Fprintln(w, "ID\tTITLE\tPRIORITY\tSTATUS\tTAGS")
    fmt.Fprintln(w, "--\t-----\t--------\t------\t----")

    for _, task := range tasks {
        // Format tags as comma-separated
        tags := strings.Join(getTagNames(task.Tags), ", ")

        // Colorize status
        statusColor := getStatusColor(task.Status)

        fmt.Fprintf(w, "%d\t%s\t%s\t%s\t%s\n",
            task.ID,
            task.Title,
            task.Priority,
            statusColor.Sprint(task.Status),
            tags,
        )
    }

    w.Flush()
}
```

**Color Coding**:
```go
func getStatusColor(status Status) *color.Color {
    switch status {
    case StatusCompleted: return color.New(color.FgGreen)
    case StatusInProgress: return color.New(color.FgYellow)
    case StatusCancelled: return color.New(color.FgRed)
    default: return color.New(color.FgWhite)
    }
}
```

#### `internal/commands/complete.go` (Complete Command)

**Purpose**: Mark task as completed

```go
func completeCmd() *cobra.Command {
    return &cobra.Command{
        Use:   "complete [task ID]",
        Short: "Mark a task as completed",
        Args:  cobra.ExactArgs(1),
        RunE: func(cmd *cobra.Command, args []string) error {
            id, err := strconv.ParseUint(args[0], 10, 32)
            if err != nil {
                return fmt.Errorf("invalid task ID: %s", args[0])
            }

            if err := taskService.CompleteTask(uint(id)); err != nil {
                return fmt.Errorf("failed to complete task: %w", err)
            }

            color.Green("✓ Task marked as completed!")
            return nil
        },
    }
}
```

**Key Points**:
- Positional argument (not a flag)
- Input validation (parse to uint)
- Error wrapping with context

#### `internal/commands/update.go` (Update Command)

**Purpose**: Modify task attributes

```go
func updateCmd() *cobra.Command {
    var title, description string
    var priority, status int

    cmd := &cobra.Command{
        Use:   "update [task ID]",
        Short: "Update a task",
        Args:  cobra.ExactArgs(1),
        RunE: func(cmd *cobra.Command, args []string) error {
            id, err := strconv.ParseUint(args[0], 10, 32)
            if err != nil {
                return fmt.Errorf("invalid task ID: %s", args[0])
            }

            // Build updates map
            updates := make(map[string]interface{})
            if title != "" {
                updates["title"] = title
            }
            if description != "" {
                updates["description"] = description
            }
            if priority >= 0 {
                updates["priority"] = Priority(priority)
            }
            if status >= 0 {
                updates["status"] = Status(status)
            }

            if len(updates) == 0 {
                return fmt.Errorf("no updates specified")
            }

            return taskService.UpdateTask(uint(id), updates)
        },
    }

    cmd.Flags().StringVarP(&title, "title", "t", "", "New title")
    cmd.Flags().StringVarP(&description, "description", "d", "", "New description")
    cmd.Flags().IntVarP(&priority, "priority", "p", -1, "Priority (0-3)")
    cmd.Flags().IntVarP(&status, "status", "s", -1, "Status (0-3)")

    return cmd
}
```

**Design**:
- Use -1 as "not provided" sentinel value
- Build map of only changed fields
- Validate at least one field changed

#### `internal/commands/delete.go` (Delete Command)

**Purpose**: Remove a task

```go
func deleteCmd() *cobra.Command {
    return &cobra.Command{
        Use:   "delete [task ID]",
        Short: "Delete a task",
        Args:  cobra.ExactArgs(1),
        RunE: func(cmd *cobra.Command, args []string) error {
            id, err := strconv.ParseUint(args[0], 10, 32)
            if err != nil {
                return fmt.Errorf("invalid task ID: %s", args[0])
            }

            if err := taskService.DeleteTask(uint(id)); err != nil {
                return fmt.Errorf("failed to delete task: %w", err)
            }

            color.Red("✓ Task deleted.")
            return nil
        },
    }
}
```

#### `internal/commands/stats.go` (Statistics Command)

**Purpose**: Display task statistics dashboard

```go
func statsCmd() *cobra.Command {
    return &cobra.Command{
        Use:   "stats",
        Short: "Display task statistics",
        RunE: func(cmd *cobra.Command, args []string) error {
            stats, err := taskService.GetStatistics()
            if err != nil {
                return fmt.Errorf("failed to get statistics: %w", err)
            }

            fmt.Println("Task Statistics")
            fmt.Println("===============")
            fmt.Printf("Total Tasks:     %d\n", stats.Total)

            // Calculate completion rate
            if stats.Total > 0 {
                rate := float64(stats.Completed) / float64(stats.Total) * 100
                color.Green("Completed:       %d (%.1f%%)", stats.Completed, rate)
            } else {
                fmt.Printf("Completed:       %d\n", stats.Completed)
            }

            fmt.Printf("Pending:         %d\n", stats.Pending)
            fmt.Printf("In Progress:     %d\n", stats.InProgress)
            fmt.Printf("Cancelled:       %d\n", stats.Cancelled)

            return nil
        },
    }
}
```

**Features**:
- Percentage calculation
- Colorized completion rate
- Formatted output

## Implementation Guide

This section provides a step-by-step guide to building TaskMaster from scratch.

### Step 1: Project Setup

```bash
# Create project directory
mkdir taskmaster
cd taskmaster

# Initialize Go module
go mod init github.com/yourusername/taskmaster

# Create directory structure
mkdir -p cmd/taskmaster
mkdir -p internal/{models,repository,service,commands}
mkdir -p tests
```

### Step 2: Define Domain Models

Create `internal/models/task.go`:

```go
package models

import (
    "time"
    "gorm.io/gorm"
)

// Start with basic Task struct
type Task struct {
    ID          uint           `gorm:"primarykey"`
    Title       string         `gorm:"not null"`
    Description string
    CreatedAt   time.Time
    UpdatedAt   time.Time
}
```

**Why start simple?**
- Verify database connectivity first
- Add complexity incrementally
- Easier to debug issues

### Step 3: Create Repository Layer

Create `internal/repository/task_repository.go`:

```go
package repository

import (
    "github.com/yourusername/taskmaster/internal/models"
    "gorm.io/gorm"
)

type TaskRepository interface {
    Create(task *models.Task) error
    GetByID(id uint) (*models.Task, error)
}

type taskRepository struct {
    db *gorm.DB
}

func NewTaskRepository(db *gorm.DB) TaskRepository {
    return &taskRepository{db: db}
}

func (r *taskRepository) Create(task *models.Task) error {
    return r.db.Create(task).Error
}

func (r *taskRepository) GetByID(id uint) (*models.Task, error) {
    var task models.Task
    err := r.db.First(&task, id).Error
    return &task, err
}
```

**Test it**:
```go
// In tests/repository_test.go
func TestCreate(t *testing.T) {
    db := setupTestDB(t)
    repo := repository.NewTaskRepository(db)

    task := &models.Task{Title: "Test"}
    err := repo.Create(task)
    assert.NoError(t, err)
    assert.NotZero(t, task.ID)
}
```

### Step 4: Add Service Layer

Create `internal/service/task_service.go`:

```go
package service

import (
    "errors"
    "strings"
    "github.com/yourusername/taskmaster/internal/models"
    "github.com/yourusername/taskmaster/internal/repository"
)

var ErrEmptyTitle = errors.New("task title cannot be empty")

type TaskService interface {
    CreateTask(title, description string) (*models.Task, error)
}

type taskService struct {
    repo repository.TaskRepository
}

func NewTaskService(repo repository.TaskRepository) TaskService {
    return &taskService{repo: repo}
}

func (s *taskService) CreateTask(title, description string) (*models.Task, error) {
    title = strings.TrimSpace(title)
    if title == "" {
        return nil, ErrEmptyTitle
    }

    task := &models.Task{
        Title:       title,
        Description: strings.TrimSpace(description),
    }

    if err := s.repo.Create(task); err != nil {
        return nil, err
    }

    return task, nil
}
```

### Step 5: Build CLI with Cobra

Install Cobra:
```bash
go get -u github.com/spf13/cobra
```

Create `internal/commands/root.go`:

```go
package commands

import (
    "github.com/spf13/cobra"
    "github.com/yourusername/taskmaster/internal/service"
)

var taskService service.TaskService

func Execute(svc service.TaskService) error {
    taskService = svc

    rootCmd := &cobra.Command{
        Use:   "taskmaster",
        Short: "A CLI task management tool",
    }

    rootCmd.AddCommand(addCmd())

    return rootCmd.Execute()
}
```

Create `internal/commands/add.go`:

```go
package commands

import (
    "fmt"
    "github.com/spf13/cobra"
)

func addCmd() *cobra.Command {
    var title, description string

    cmd := &cobra.Command{
        Use:   "add",
        Short: "Add a new task",
        RunE: func(cmd *cobra.Command, args []string) error {
            task, err := taskService.CreateTask(title, description)
            if err != nil {
                return err
            }
            fmt.Printf("Created task #%d: %s\n", task.ID, task.Title)
            return nil
        },
    }

    cmd.Flags().StringVarP(&title, "title", "t", "", "Task title (required)")
    cmd.Flags().StringVarP(&description, "description", "d", "", "Description")
    cmd.MarkFlagRequired("title")

    return cmd
}
```

### Step 6: Wire Everything Together

Create `cmd/taskmaster/main.go`:

```go
package main

import (
    "fmt"
    "os"
    "path/filepath"

    "github.com/yourusername/taskmaster/internal/commands"
    "github.com/yourusername/taskmaster/internal/models"
    "github.com/yourusername/taskmaster/internal/repository"
    "github.com/yourusername/taskmaster/internal/service"
    "gorm.io/driver/sqlite"
    "gorm.io/gorm"
)

func main() {
    if err := run(); err != nil {
        fmt.Fprintf(os.Stderr, "Error: %v\n", err)
        os.Exit(1)
    }
}

func run() error {
    // Setup database path
    home, err := os.UserHomeDir()
    if err != nil {
        return err
    }

    dataDir := filepath.Join(home, ".taskmaster")
    dbPath := filepath.Join(dataDir, "tasks.db")

    // Create directory
    if err := os.MkdirAll(dataDir, 0755); err != nil {
        return err
    }

    // Open database
    db, err := gorm.Open(sqlite.Open(dbPath), &gorm.Config{})
    if err != nil {
        return err
    }

    // Migrate
    if err := db.AutoMigrate(&models.Task{}); err != nil {
        return err
    }

    // Build layers
    repo := repository.NewTaskRepository(db)
    svc := service.NewTaskService(repo)

    // Execute
    return commands.Execute(svc)
}
```

### Step 7: Test the MVP

```bash
# Build
go build -o taskmaster cmd/taskmaster/main.go

# Test
./taskmaster add -t "My first task"
# Output: Created task #1: My first task

# Verify database
sqlite3 ~/.taskmaster/tasks.db "SELECT * FROM tasks;"
# Output: 1|My first task||2024-01-01 12:00:00|2024-01-01 12:00:00
```

### Step 8: Add Priority and Status

Update `internal/models/task.go`:

```go
type Priority int

const (
    PriorityLow Priority = iota
    PriorityMedium
    PriorityHigh
    PriorityCritical
)

func (p Priority) String() string {
    return [...]string{"Low", "Medium", "High", "Critical"}[p]
}

type Status int

const (
    StatusPending Status = iota
    StatusInProgress
    StatusCompleted
    StatusCancelled
)

func (s Status) String() string {
    return [...]string{"Pending", "In Progress", "Completed", "Cancelled"}[s]
}

// Add to Task struct:
Priority    Priority       `gorm:"default:1"`
Status      Status         `gorm:"default:0"`
```

Update service to accept priority:

```go
func (s *taskService) CreateTask(title, description string, priority Priority) (*Task, error) {
    // ... validation ...
    task := &Task{
        Title:       title,
        Description: description,
        Priority:    priority,
        Status:      StatusPending,
    }
    // ...
}
```

### Step 9: Add Tags (Many-to-Many)

Add Tag model:

```go
type Tag struct {
    ID   uint   `gorm:"primarykey"`
    Name string `gorm:"unique;not null"`
}

// Update Task:
Tags []Tag `gorm:"many2many:task_tags;"`
```

Update repository Create to handle tags:

```go
func (r *taskRepository) Create(task *Task) error {
    // GORM automatically handles the many-to-many relationship
    return r.db.Create(task).Error
}
```

Update service:

```go
func (s *taskService) CreateTask(title, description string, priority Priority, tags []string) (*Task, error) {
    // ... validation ...

    task := &Task{
        Title:    title,
        // ...
    }

    for _, tagName := range tags {
        tagName = strings.TrimSpace(tagName)
        if tagName != "" {
            task.Tags = append(task.Tags, Tag{Name: tagName})
        }
    }

    return s.repo.Create(task)
}
```

### Step 10: Add List Command

Create `internal/commands/list.go`:

```go
func listCmd() *cobra.Command {
    return &cobra.Command{
        Use:   "list",
        Short: "List tasks",
        RunE: func(cmd *cobra.Command, args []string) error {
            tasks, err := taskService.ListTasks(TaskFilter{})
            if err != nil {
                return err
            }

            for _, task := range tasks {
                fmt.Printf("#%d: %s [%s]\n", task.ID, task.Title, task.Status)
            }
            return nil
        },
    }
}
```

Add to repository:

```go
func (r *taskRepository) List(filter TaskFilter) ([]*Task, error) {
    var tasks []*Task
    query := r.db.Preload("Tags")

    // Add filters later

    err := query.Order("created_at DESC").Find(&tasks).Error
    return tasks, err
}
```

### Step 11: Add Filtering

Update repository's List method:

```go
func (r *taskRepository) List(filter TaskFilter) ([]*Task, error) {
    var tasks []*Task
    query := r.db.Preload("Tags")

    if filter.Status != nil {
        query = query.Where("status = ?", *filter.Status)
    }

    if filter.Priority != nil {
        query = query.Where("priority = ?", *filter.Priority)
    }

    if filter.Tag != "" {
        query = query.Joins("JOIN task_tags ON task_tags.task_id = tasks.id").
                      Joins("JOIN tags ON tags.id = task_tags.tag_id").
                      Where("tags.name = ?", filter.Tag)
    }

    if filter.Search != "" {
        pattern := "%" + filter.Search + "%"
        query = query.Where("title LIKE ? OR description LIKE ?", pattern, pattern)
    }

    err := query.Order("priority DESC, created_at DESC").Find(&tasks).Error
    return tasks, err
}
```

Update list command:

```go
func listCmd() *cobra.Command {
    var status, priority, tag, search string

    cmd := &cobra.Command{
        Use:   "list",
        Short: "List tasks",
        RunE: func(cmd *cobra.Command, args []string) error {
            filter := buildFilter(status, priority, tag, search)
            tasks, err := taskService.ListTasks(filter)
            // ...
        },
    }

    cmd.Flags().StringVar(&status, "status", "", "Filter by status")
    cmd.Flags().StringVar(&priority, "priority", "", "Filter by priority")
    cmd.Flags().StringVar(&tag, "tag", "", "Filter by tag")
    cmd.Flags().StringVar(&search, "search", "", "Search text")

    return cmd
}
```

### Step 12: Add Color Output

Install color package:

```bash
go get -u github.com/fatih/color
```

Update list command:

```go
import "github.com/fatih/color"

func displayTasks(tasks []*Task) {
    for _, task := range tasks {
        statusColor := getStatusColor(task.Status)
        fmt.Printf("#%d: %s [%s]\n",
            task.ID,
            task.Title,
            statusColor.Sprint(task.Status))
    }
}

func getStatusColor(status Status) *color.Color {
    switch status {
    case StatusCompleted: return color.New(color.FgGreen)
    case StatusInProgress: return color.New(color.FgYellow)
    case StatusCancelled: return color.New(color.FgRed)
    default: return color.New(color.FgWhite)
    }
}
```

### Step 13: Add Remaining Commands

Follow the same pattern for:
- `complete.go` - Update status to completed
- `update.go` - Partial updates
- `delete.go` - Soft delete
- `stats.go` - Aggregation queries

### Step 14: Write Tests

Create `tests/integration_test.go`:

```go
func TestTaskLifecycle(t *testing.T) {
    db := setupTestDB(t)
    repo := repository.NewTaskRepository(db)
    svc := service.NewTaskService(repo)

    // Create
    task, err := svc.CreateTask("Test", "", PriorityHigh, nil)
    require.NoError(t, err)

    // Read
    retrieved, err := svc.GetTask(task.ID)
    require.NoError(t, err)
    assert.Equal(t, "Test", retrieved.Title)

    // Update (complete)
    err = svc.CompleteTask(task.ID)
    require.NoError(t, err)

    // Verify
    completed, err := svc.GetTask(task.ID)
    assert.Equal(t, StatusCompleted, completed.Status)

    // Delete
    err = svc.DeleteTask(task.ID)
    require.NoError(t, err)

    _, err = svc.GetTask(task.ID)
    assert.Error(t, err)
}
```

### Step 15: Create Dockerfile

Multi-stage build for efficiency:

```dockerfile
FROM golang:1.23-alpine AS builder

WORKDIR /app
COPY go.mod go.sum ./
RUN go mod download

COPY . .
RUN CGO_ENABLED=1 GOOS=linux go build -o taskmaster cmd/taskmaster/main.go

FROM alpine:latest
RUN apk --no-cache add ca-certificates sqlite

WORKDIR /root/
COPY --from=builder /app/taskmaster .
RUN mkdir -p /root/.taskmaster

ENTRYPOINT ["./taskmaster"]
```

### Step 16: Add Makefile

```makefile
.PHONY: build test run clean

build:
	go build -o bin/taskmaster cmd/taskmaster/main.go

test:
	go test -v ./...

run:
	go run cmd/taskmaster/main.go

clean:
	rm -rf bin/
```

## Source Code Walkthrough

### Complete main.go Analysis

```go
package main

import (
    "fmt"
    "os"
    "path/filepath"

    "github.com/yourusername/taskmaster/internal/commands"
    "github.com/yourusername/taskmaster/internal/models"
    "github.com/yourusername/taskmaster/internal/repository"
    "github.com/yourusername/taskmaster/internal/service"
    "gorm.io/driver/sqlite"
    "gorm.io/gorm"
)

// main is the application entry point
// It delegates all logic to run() for better error handling
func main() {
    if err := run(); err != nil {
        fmt.Fprintf(os.Stderr, "Error: %v\n", err)
        os.Exit(1)
    }
}

// run contains the actual application logic
// Returns error instead of calling os.Exit for testability
func run() error {
    // Step 1: Determine where to store the database
    // Use user's home directory for portable location
    home, err := os.UserHomeDir()
    if err != nil {
        return fmt.Errorf("failed to get home directory: %w", err)
    }

    // Step 2: Build paths
    // .taskmaster/ directory in user's home
    // Contains tasks.db SQLite database file
    dataDir := filepath.Join(home, ".taskmaster")
    dbPath := filepath.Join(dataDir, "tasks.db")

    // Step 3: Ensure directory exists
    // MkdirAll creates parent directories if needed
    // 0755 = rwxr-xr-x permissions
    if err := os.MkdirAll(dataDir, 0755); err != nil {
        return fmt.Errorf("failed to create data directory: %w", err)
    }

    // Step 4: Open database connection
    // GORM handles connection pooling automatically
    db, err := gorm.Open(sqlite.Open(dbPath), &gorm.Config{})
    if err != nil {
        return fmt.Errorf("failed to open database: %w", err)
    }

    // Step 5: Run migrations
    // AutoMigrate creates/updates tables based on struct definitions
    // Safe to run on every startup (no-op if schema matches)
    if err := db.AutoMigrate(&models.Task{}, &models.Tag{}); err != nil {
        return fmt.Errorf("failed to migrate database: %w", err)
    }

    // Step 6: Initialize layers (dependency injection)
    // Repository layer depends on database
    repo := repository.NewTaskRepository(db)

    // Service layer depends on repository
    svc := service.NewTaskService(repo)

    // Step 7: Execute CLI
    // Commands layer depends on service
    // Cobra handles flag parsing and command routing
    return commands.Execute(svc)
}
```

**Key Design Decisions**:

1. **Error Handling**: Use `run()` function to return errors instead of calling `os.Exit()` directly
   - Allows testing of main logic
   - Consistent error handling pattern
   - Error wrapping with `%w` for error chains

2. **Path Management**: Use `filepath.Join()` for cross-platform compatibility
   - Works on Windows, Linux, macOS
   - Handles path separators correctly

3. **Database Location**: Store in `~/.taskmaster/`
   - Standard location for user data
   - No system-wide installation needed
   - Easy to backup or reset

4. **Migrations**: Run on every startup
   - Ensures database schema is up-to-date
   - No separate migration command needed
   - GORM makes this safe and fast

5. **Dependency Injection**: Build layer chain explicitly
   - Clear dependency graph
   - Easy to test with mocks
   - No global state or singletons

### Complete Task Model Analysis

```go
package models

import (
    "time"
    "gorm.io/gorm"
)

// Task represents a task item in the system
// Uses GORM struct tags for ORM mapping
type Task struct {
    // Primary key - auto-incrementing unsigned integer
    ID uint `gorm:"primarykey" json:"id"`

    // Required field - database constraint enforced
    Title string `gorm:"not null" json:"title"`

    // Optional description
    Description string `json:"description"`

    // Priority level with default value (Medium = 1)
    // Using integer type for efficient storage and sorting
    Priority Priority `gorm:"default:1" json:"priority"`

    // Task status with default (Pending = 0)
    Status Status `gorm:"default:0" json:"status"`

    // Optional due date (pointer allows NULL in database)
    DueDate *time.Time `json:"due_date,omitempty"`

    // Many-to-many relationship with tags
    // Junction table: task_tags
    Tags []Tag `gorm:"many2many:task_tags;" json:"tags"`

    // Auto-managed timestamps by GORM
    CreatedAt time.Time `json:"created_at"`
    UpdatedAt time.Time `json:"updated_at"`

    // Soft delete support - sets timestamp instead of removing row
    // Index for query performance
    // json:"-" excludes from JSON serialization
    DeletedAt gorm.DeletedAt `gorm:"index" json:"-"`
}

// Priority represents task priority levels
// Using iota for auto-incrementing constants
// Integer type for efficient storage (1 byte)
type Priority int

const (
    PriorityLow      Priority = 0  // Can wait
    PriorityMedium   Priority = 1  // Normal importance (default)
    PriorityHigh     Priority = 2  // Important
    PriorityCritical Priority = 3  // Urgent, do first
)

// String implements fmt.Stringer interface
// Enables: fmt.Println(task.Priority) → "High"
func (p Priority) String() string {
    return [...]string{"Low", "Medium", "High", "Critical"}[p]
}

// Status represents task lifecycle states
type Status int

const (
    StatusPending    Status = 0  // Not started (default)
    StatusInProgress Status = 1  // Currently working on
    StatusCompleted  Status = 2  // Finished
    StatusCancelled  Status = 3  // Abandoned
)

// String implementation for Status
func (s Status) String() string {
    return [...]string{"Pending", "In Progress", "Completed", "Cancelled"}[s]
}

// Tag represents a category or label
// Many-to-many relationship with tasks
type Tag struct {
    ID   uint   `gorm:"primarykey"`
    Name string `gorm:"unique;not null"`  // Unique constraint prevents duplicates
}

// TaskFilter encapsulates query parameters for filtering tasks
// Used by repository layer to build dynamic queries
type TaskFilter struct {
    // Pointer types distinguish "not set" from "zero value"
    // nil = no filter, &value = filter by value
    Status   *Status
    Priority *Priority

    // Empty string = no filter
    Tag    string
    Search string  // Full-text search in title/description
}
```

**Design Decisions Explained**:

1. **Integer Enums vs Strings**:
   - Store as integers (1 byte) vs strings (variable length)
   - Faster sorting and comparison
   - Type-safe with const blocks
   - Human-readable with String() methods

2. **Pointer Fields**:
   - `DueDate *time.Time`: NULL in database if not set
   - `TaskFilter.Status *Status`: Distinguish between "no filter" and "filter by pending (0)"
   - Without pointers, zero values are ambiguous

3. **GORM Tags**:
   - `primarykey`: Enables auto-increment
   - `not null`: Database constraint
   - `default:X`: Default value in database
   - `many2many:table_name`: Defines junction table
   - `index`: Creates database index for performance
   - `unique`: Enforces uniqueness constraint

4. **JSON Tags**:
   - Enable JSON serialization for potential API
   - `omitempty`: Exclude null/empty fields from JSON
   - `json:"-"`: Never include in JSON

5. **Soft Delete**:
   - `DeletedAt gorm.DeletedAt`: GORM automatically filters deleted records
   - Can recover deleted tasks if needed
   - Audit trail (see what was deleted and when)

6. **Timestamps**:
   - `CreatedAt`, `UpdatedAt`: GORM manages automatically
   - No manual setting needed
   - Audit trail built-in

## Testing Strategy

### Testing Pyramid

```
        /\
       /  \
      / E2E \             ← Few, slow, comprehensive
     /______\
    /        \
   / Integration\         ← Some, medium speed
  /____________\
 /              \
/   Unit Tests   \        ← Many, fast, focused
/________________\
```

### Unit Tests (Fast, Isolated)

**Service Layer Tests**:

```go
package service_test

import (
    "testing"
    "github.com/stretchr/testify/assert"
    "github.com/stretchr/testify/mock"
)

// Mock repository for unit testing service
type MockRepository struct {
    mock.Mock
}

func (m *MockRepository) Create(task *Task) error {
    args := m.Called(task)
    return args.Error(0)
}

func TestCreateTask_Validation(t *testing.T) {
    mockRepo := new(MockRepository)
    svc := service.NewTaskService(mockRepo)

    tests := []struct {
        name        string
        title       string
        expectError bool
    }{
        {"Valid title", "Fix bug", false},
        {"Empty title", "", true},
        {"Whitespace only", "   ", true},
    }

    for _, tt := range tests {
        t.Run(tt.name, func(t *testing.T) {
            _, err := svc.CreateTask(tt.title, "", PriorityMedium, nil)
            if tt.expectError {
                assert.Error(t, err)
            } else {
                assert.NoError(t, err)
            }
        })
    }
}

func TestCreateTask_SanitizesInput(t *testing.T) {
    mockRepo := new(MockRepository)
    svc := service.NewTaskService(mockRepo)

    // Expect repository to receive trimmed title
    mockRepo.On("Create", mock.MatchedBy(func(task *Task) bool {
        return task.Title == "Trimmed"
    })).Return(nil)

    _, err := svc.CreateTask("  Trimmed  ", "", PriorityMedium, nil)
    assert.NoError(t, err)
    mockRepo.AssertExpectations(t)
}
```

### Integration Tests (Medium, Real Database)

**Full Stack Tests**:

```go
package tests

import (
    "testing"
    "github.com/stretchr/testify/assert"
    "github.com/stretchr/testify/require"
    "gorm.io/driver/sqlite"
    "gorm.io/gorm"
)

// setupTestDB creates an in-memory SQLite database for testing
// Fast and isolated - each test gets a fresh database
func setupTestDB(t *testing.T) *gorm.DB {
    db, err := gorm.Open(sqlite.Open(":memory:"), &gorm.Config{})
    require.NoError(t, err, "Failed to open test database")

    err = db.AutoMigrate(&models.Task{}, &models.Tag{})
    require.NoError(t, err, "Failed to migrate test database")

    return db
}

// TestTaskLifecycle tests the complete CRUD cycle
func TestTaskLifecycle(t *testing.T) {
    // Setup
    db := setupTestDB(t)
    repo := repository.NewTaskRepository(db)
    svc := service.NewTaskService(repo)

    // CREATE
    task, err := svc.CreateTask(
        "Test Task",
        "Test Description",
        models.PriorityHigh,
        []string{"test", "integration"},
    )
    require.NoError(t, err, "Create should succeed")
    assert.NotZero(t, task.ID, "ID should be set")
    assert.Equal(t, "Test Task", task.Title)
    assert.Equal(t, models.StatusPending, task.Status)
    assert.Len(t, task.Tags, 2, "Should have 2 tags")

    // READ
    retrieved, err := svc.GetTask(task.ID)
    require.NoError(t, err, "Get should succeed")
    assert.Equal(t, task.ID, retrieved.ID)
    assert.Equal(t, "Test Task", retrieved.Title)

    // UPDATE (Complete)
    err = svc.CompleteTask(task.ID)
    require.NoError(t, err, "Complete should succeed")

    completed, err := svc.GetTask(task.ID)
    require.NoError(t, err)
    assert.Equal(t, models.StatusCompleted, completed.Status)

    // UPDATE (Modify)
    err = svc.UpdateTask(task.ID, map[string]interface{}{
        "title": "Updated Title",
    })
    require.NoError(t, err)

    updated, err := svc.GetTask(task.ID)
    require.NoError(t, err)
    assert.Equal(t, "Updated Title", updated.Title)

    // DELETE
    err = svc.DeleteTask(task.ID)
    require.NoError(t, err, "Delete should succeed")

    _, err = svc.GetTask(task.ID)
    assert.ErrorIs(t, err, repository.ErrTaskNotFound, "Should not find deleted task")
}

// TestTaskFiltering tests all filter combinations
func TestTaskFiltering(t *testing.T) {
    db := setupTestDB(t)
    repo := repository.NewTaskRepository(db)
    svc := service.NewTaskService(repo)

    // Create test data
    tasks := []struct {
        title    string
        priority models.Priority
        tags     []string
    }{
        {"High priority urgent", models.PriorityHigh, []string{"urgent"}},
        {"Low priority later", models.PriorityLow, []string{"later"}},
        {"High priority bug", models.PriorityHigh, []string{"bug"}},
    }

    for _, tc := range tasks {
        _, err := svc.CreateTask(tc.title, "", tc.priority, tc.tags)
        require.NoError(t, err)
    }

    tests := []struct {
        name          string
        filter        models.TaskFilter
        expectedCount int
    }{
        {
            name:          "All tasks",
            filter:        models.TaskFilter{},
            expectedCount: 3,
        },
        {
            name: "High priority only",
            filter: models.TaskFilter{
                Priority: ptrPriority(models.PriorityHigh),
            },
            expectedCount: 2,
        },
        {
            name: "Tag: urgent",
            filter: models.TaskFilter{
                Tag: "urgent",
            },
            expectedCount: 1,
        },
        {
            name: "Search: bug",
            filter: models.TaskFilter{
                Search: "bug",
            },
            expectedCount: 1,
        },
    }

    for _, tt := range tests {
        t.Run(tt.name, func(t *testing.T) {
            results, err := svc.ListTasks(tt.filter)
            require.NoError(t, err)
            assert.Len(t, results, tt.expectedCount)
        })
    }
}

// Helper function to create priority pointers
func ptrPriority(p models.Priority) *models.Priority {
    return &p
}

// TestConcurrentAccess tests thread safety
func TestConcurrentAccess(t *testing.T) {
    db := setupTestDB(t)
    repo := repository.NewTaskRepository(db)
    svc := service.NewTaskService(repo)

    const goroutines = 10
    const tasksPerGoroutine = 10

    done := make(chan bool)

    for i := 0; i < goroutines; i++ {
        go func(id int) {
            for j := 0; j < tasksPerGoroutine; j++ {
                title := fmt.Sprintf("Task %d-%d", id, j)
                _, err := svc.CreateTask(title, "", models.PriorityMedium, nil)
                assert.NoError(t, err)
            }
            done <- true
        }(i)
    }

    for i := 0; i < goroutines; i++ {
        <-done
    }

    tasks, err := svc.ListTasks(models.TaskFilter{})
    require.NoError(t, err)
    assert.Len(t, tasks, goroutines*tasksPerGoroutine)
}
```

### Running Tests

```bash
# Run all tests
go test ./...

# Run with coverage
go test -coverprofile=coverage.out ./...
go tool cover -html=coverage.out

# Run with race detector
go test -race ./...

# Run specific test
go test -run TestTaskLifecycle ./tests

# Verbose output
go test -v ./...

# Benchmark (if you add benchmarks)
go test -bench=. ./...
```

### Coverage Goals

- **Overall**: 80%+
- **Service Layer**: 90%+ (critical business logic)
- **Repository Layer**: 85%+ (data access)
- **Commands Layer**: 70%+ (harder to test, more UI-focused)

### Test Organization

```
tests/
├── integration_test.go     # Full stack tests
│
internal/
├── service/
│   ├── task_service.go
│   └── task_service_test.go  # Unit tests for service
│
├── repository/
│   ├── task_repository.go
│   └── task_repository_test.go  # Unit tests for repo
│
└── commands/
    ├── add.go
    └── add_test.go  # CLI tests (if needed)
```

## Deployment Guide

### Local Development

```bash
# Install dependencies
go mod download

# Run in development mode (direct execution)
go run cmd/taskmaster/main.go add -t "Test task"

# Build optimized binary
go build -ldflags="-s -w" -o taskmaster cmd/taskmaster/main.go

# Install system-wide
sudo mv taskmaster /usr/local/bin/
```

### Docker Deployment

#### Building the Image

```bash
# Build with Docker
docker build -t taskmaster:latest .

# Build with specific tag
docker build -t taskmaster:v1.0.0 .

# Build for multiple platforms (using buildx)
docker buildx build --platform linux/amd64,linux/arm64 -t taskmaster:latest .
```

#### Running with Docker

```bash
# One-off command
docker run -it --rm taskmaster:latest list

# With persistent storage (named volume)
docker volume create taskmaster-data
docker run -it --rm \
    -v taskmaster-data:/root/.taskmaster \
    taskmaster:latest add -t "Docker task"

# Create shell alias for convenience
alias tm='docker run -it --rm -v taskmaster-data:/root/.taskmaster taskmaster:latest'

# Use like native command
tm add -t "My task" -p high
tm list
tm complete 1
```

#### Docker Compose

Create `docker-compose.yml`:

```yaml
version: '3.8'

services:
  taskmaster:
    build: .
    image: taskmaster:latest
    volumes:
      - taskmaster-data:/root/.taskmaster
    stdin_open: true
    tty: true

volumes:
  taskmaster-data:
```

Usage:

```bash
# Build and start
docker-compose up -d

# Run commands
docker-compose run --rm taskmaster add -t "Task from compose"
docker-compose run --rm taskmaster list

# Stop
docker-compose down
```

### Production Deployment

#### Binary Distribution

```bash
# Build for Linux (from any OS)
GOOS=linux GOARCH=amd64 go build -o taskmaster-linux-amd64 cmd/taskmaster/main.go

# Build for macOS
GOOS=darwin GOARCH=amd64 go build -o taskmaster-darwin-amd64 cmd/taskmaster/main.go

# Build for Windows
GOOS=windows GOARCH=amd64 go build -o taskmaster-windows-amd64.exe cmd/taskmaster/main.go

# Create release tarball
tar -czf taskmaster-v1.0.0-linux-amd64.tar.gz taskmaster-linux-amd64
```

#### Container Registry

```bash
# Tag for registry
docker tag taskmaster:latest registry.example.com/taskmaster:v1.0.0

# Push to registry
docker push registry.example.com/taskmaster:v1.0.0

# Pull and run on any machine
docker pull registry.example.com/taskmaster:v1.0.0
docker run -it registry.example.com/taskmaster:v1.0.0 list
```

#### Cloud Platforms

**AWS ECS/Fargate**:

```bash
# Build and push to ECR
aws ecr get-login-password --region us-east-1 | docker login --username AWS --password-stdin <account-id>.dkr.ecr.us-east-1.amazonaws.com

docker tag taskmaster:latest <account-id>.dkr.ecr.us-east-1.amazonaws.com/taskmaster:latest
docker push <account-id>.dkr.ecr.us-east-1.amazonaws.com/taskmaster:latest

# Create task definition and service via AWS Console or CLI
```

**Google Cloud Run**:

```bash
# Build and push to GCR
docker tag taskmaster:latest gcr.io/<project-id>/taskmaster:latest
docker push gcr.io/<project-id>/taskmaster:latest

# Deploy
gcloud run deploy taskmaster \
    --image gcr.io/<project-id>/taskmaster:latest \
    --platform managed \
    --region us-central1
```

### Production Checklist

- [ ] Set proper resource limits (memory, CPU)
- [ ] Use external volume for database persistence
- [ ] Implement backup strategy for SQLite database
- [ ] Set up monitoring and alerting
- [ ] Use multi-stage Docker builds to minimize image size
- [ ] Run as non-root user in container (security)
- [ ] Enable Docker health checks
- [ ] Use secrets management for sensitive data
- [ ] Implement log aggregation
- [ ] Set up CI/CD pipeline for automated deployments

### Database Backup

```bash
# Backup SQLite database
cp ~/.taskmaster/tasks.db ~/.taskmaster/tasks.db.backup

# Scheduled backup (cron job)
0 2 * * * cp ~/.taskmaster/tasks.db ~/.taskmaster/tasks.db.$(date +\%Y\%m\%d)

# Docker volume backup
docker run --rm \
    -v taskmaster-data:/data \
    -v $(pwd):/backup \
    alpine tar czf /backup/taskmaster-backup.tar.gz /data
```

### Monitoring

Add health check to Dockerfile:

```dockerfile
HEALTHCHECK --interval=30s --timeout=3s --start-period=5s --retries=3 \
    CMD ["./taskmaster", "list"] || exit 1
```

## Development Workflow

### Setting Up Development Environment

```bash
# Clone repository
git clone https://github.com/yourusername/taskmaster.git
cd taskmaster

# Install dependencies
go mod download

# Install development tools
go install github.com/golangci/golangci-lint/cmd/golangci-lint@latest
go install github.com/securego/gosec/v2/cmd/gosec@latest

# Run tests to verify setup
go test ./...
```

### Code Quality

```bash
# Format code
go fmt ./...

# Run linter
golangci-lint run

# Security scan
gosec ./...

# Vet code
go vet ./...
```

### Git Workflow

```bash
# Create feature branch
git checkout -b feature/add-due-dates

# Make changes and commit
git add .
git commit -m "feat: add due date support to tasks"

# Push and create PR
git push origin feature/add-due-dates
```

### Debugging

```bash
# Run with verbose logging
go run cmd/taskmaster/main.go -v add -t "Test"

# Use Delve debugger
go install github.com/go-delve/delve/cmd/dlv@latest
dlv debug cmd/taskmaster/main.go -- add -t "Test"

# Print SQL queries (add to main.go)
db, err := gorm.Open(sqlite.Open(dbPath), &gorm.Config{
    Logger: logger.Default.LogMode(logger.Info),
})
```

## Advanced Usage

### Scripting with TaskMaster

```bash
#!/bin/bash

# Bulk add tasks from file
while IFS= read -r line; do
    taskmaster add -t "$line" -p medium
done < tasks.txt

# Export tasks to JSON (would require JSON output mode)
taskmaster list --output json > tasks.json

# Weekly report
echo "Weekly Task Report"
echo "=================="
taskmaster stats
```

### Integration with Other Tools

```bash
# Add task from cron job
0 9 * * 1 taskmaster add -t "Weekly team meeting" -p high --tags meeting

# Git hook: Add task on commit
# .git/hooks/post-commit
#!/bin/bash
taskmaster add -t "Review commit $(git rev-parse --short HEAD)" --tags code-review
```

## Troubleshooting

### Common Issues

**Issue: Database locked**
```
Error: database is locked
```

**Solution**:
```bash
# Close other instances accessing the database
pkill taskmaster

# Check for stale locks
fuser ~/.taskmaster/tasks.db

# In code: Add timeout
db, err := gorm.Open(sqlite.Open(dbPath + "?_busy_timeout=5000"), ...)
```

**Issue: Permission denied**
```
Error: permission denied: ~/.taskmaster
```

**Solution**:
```bash
# Fix permissions
chmod 755 ~/.taskmaster
chmod 644 ~/.taskmaster/tasks.db
```

**Issue: Module not found**
```
Error: cannot find module github.com/yourusername/taskmaster
```

**Solution**:
```bash
# Update module path in all files
find . -name "*.go" -exec sed -i 's|github.com/yourusername/taskmaster|github.com/YOURNAME/taskmaster|g' {} +

# Update go.mod
go mod edit -module github.com/YOURNAME/taskmaster
go mod tidy
```

## Contributing

### How to Contribute

1. Fork the repository
2. Create a feature branch (`git checkout -b feature/amazing-feature`)
3. Make your changes
4. Add tests for new functionality
5. Ensure all tests pass (`go test ./...`)
6. Run linters (`golangci-lint run`)
7. Commit your changes (`git commit -m 'feat: add amazing feature'`)
8. Push to your branch (`git push origin feature/amazing-feature`)
9. Open a Pull Request

### Coding Standards

- Follow [Effective Go](https://golang.org/doc/effective_go)
- Use `gofmt` for formatting
- Write tests for new features
- Document exported functions
- Use meaningful variable names
- Keep functions small and focused

### Future Enhancements

Potential features to add:

- [ ] Task dependencies (subtasks)
- [ ] Reminders and notifications
- [ ] Export to Markdown/CSV
- [ ] Recurring tasks
- [ ] Time tracking
- [ ] Task templates
- [ ] Multi-user support
- [ ] Web UI
- [ ] Mobile app
- [ ] Cloud sync

## License

MIT License - see LICENSE file for details.

## Acknowledgments

- Built with [Cobra](https://github.com/spf13/cobra) for CLI framework
- Uses [GORM](https://gorm.io) for database ORM
- Colorized output via [fatih/color](https://github.com/fatih/color)
- Inspired by todo.txt and Taskwarrior

---

**Tutorial Reference**: This project was built following "The Modern Go Tutorial". For step-by-step explanations and learning materials, visit:
https://golang-tutorial.com/06-applied-projects/01-cli-task-manager

**Questions or Issues?** Open an issue on GitHub or reach out to the maintainers.

**Happy Task Managing!**
