package main

import (
    "fmt"
    "os"
    "path/filepath"

    "github.com/yourusername/taskmaster/internal/commands"
    "github.com/yourusername/taskmaster/internal/models"
    "github.com/yourusername/taskmaster/internal/repository"
    "github.com/yourusername/taskmaster/internal/service"
    "gorm.io/driver/sqlite"
    "gorm.io/gorm"
)

func main() {
    if err := run(); err != nil {
        fmt.Fprintf(os.Stderr, "Error: %v\n", err)
        os.Exit(1)
    }
}

func run() error {
    // Setup database path (user's home directory)
    home, err := os.UserHomeDir()
    if err != nil {
        return fmt.Errorf("failed to get home directory: %w", err)
    }

    dataDir := filepath.Join(home, ".taskmaster")
    dbPath := filepath.Join(dataDir, "tasks.db")

    // Ensure directory exists
    if err := os.MkdirAll(dataDir, 0755); err != nil {
        return fmt.Errorf("failed to create data directory: %w", err)
    }

    // Connect to database
    db, err := gorm.Open(sqlite.Open(dbPath), &gorm.Config{})
    if err != nil {
        return fmt.Errorf("failed to open database: %w", err)
    }

    // Run migrations (create/update tables)
    if err := db.AutoMigrate(&models.Task{}, &models.Tag{}); err != nil {
        return fmt.Errorf("failed to migrate database: %w", err)
    }

    // Initialize layers
    repo := repository.NewTaskRepository(db)
    svc := service.NewTaskService(repo)

    // Execute CLI
    return commands.Execute(svc)
}
