package commands

import (
    "fmt"

    "github.com/fatih/color"
    "github.com/spf13/cobra"
    "github.com/yourusername/taskmaster/internal/models"
)

func addCmd() *cobra.Command {
    var (
        title       string
        description string
        priority    string
        tags        []string
    )

    cmd := &cobra.Command{
        Use:   "add",
        Short: "Add a new task",
        Long: `Add a new task with title, description, priority, and tags.

Examples:
  # Quick task
  taskmaster add -t "Fix bug in login"

  # With all options
  taskmaster add -t "Deploy v2.0" -d "Deploy to production" -p high --tags deploy,prod`,
        RunE: func(cmd *cobra.Command, args []string) error {
            return addTask(title, description, priority, tags)
        },
    }

    cmd.Flags().StringVarP(&title, "title", "t", "", "Task title (required)")
    cmd.Flags().StringVarP(&description, "description", "d", "", "Task description")
    cmd.Flags().StringVarP(&priority, "priority", "p", "medium", "Priority (low, medium, high, critical)")
    cmd.Flags().StringSliceVar(&tags, "tags", []string{}, "Task tags (comma-separated)")
    cmd.MarkFlagRequired("title")

    return cmd
}

func addTask(title, description, priorityStr string, tags []string) error {
    priority := parsePriority(priorityStr)

    task, err := taskService.CreateTask(title, description, priority, tags)
    if err != nil {
        return fmt.Errorf("failed to create task: %w", err)
    }

    // Success output with color
    color.Green("✓ Task created successfully!")
    fmt.Printf("ID: %d\n", task.ID)
    fmt.Printf("Title: %s\n", task.Title)
    fmt.Printf("Priority: %s\n", task.Priority)

    if len(task.Tags) > 0 {
        fmt.Print("Tags: ")
        for i, tag := range task.Tags {
            if i > 0 {
                fmt.Print(", ")
            }
            fmt.Print(tag.Name)
        }
        fmt.Println()
    }

    return nil
}

func parsePriority(s string) models.Priority {
    switch s {
    case "low":
        return models.PriorityLow
    case "high":
        return models.PriorityHigh
    case "critical":
        return models.PriorityCritical
    default:
        return models.PriorityMedium
    }
}
