package commands

import (
    "fmt"
    "os"
    "text/tabwriter"

    "github.com/fatih/color"
    "github.com/spf13/cobra"
    "github.com/yourusername/taskmaster/internal/models"
)

func listCmd() *cobra.Command {
    var (
        status   string
        priority string
        tag      string
        search   string
    )

    cmd := &cobra.Command{
        Use:   "list",
        Short: "List tasks",
        Long: `List tasks with optional filtering.

Examples:
  # List all tasks
  taskmaster list

  # List completed tasks
  taskmaster list --status completed

  # List high-priority tasks
  taskmaster list --priority high

  # Search for tasks
  taskmaster list --search "bug"

  # Combine filters
  taskmaster list --status pending --priority high --tag urgent`,
        RunE: func(cmd *cobra.Command, args []string) error {
            filter := models.TaskFilter{
                Tag:    tag,
                Search: search,
            }

            if status != "" {
                s := parseStatus(status)
                filter.Status = &s
            }

            if priority != "" {
                p := parsePriority(priority)
                filter.Priority = &p
            }

            tasks, err := taskService.ListTasks(filter)
            if err != nil {
                return fmt.Errorf("failed to list tasks: %w", err)
            }

            displayTasks(tasks)
            return nil
        },
    }

    cmd.Flags().StringVarP(&status, "status", "s", "", "Filter by status (pending, progress, completed, cancelled)")
    cmd.Flags().StringVarP(&priority, "priority", "p", "", "Filter by priority (low, medium, high, critical)")
    cmd.Flags().StringVar(&tag, "tag", "", "Filter by tag")
    cmd.Flags().StringVar(&search, "search", "", "Search in title and description")

    return cmd
}

func displayTasks(tasks []*models.Task) {
    if len(tasks) == 0 {
        color.Yellow("No tasks found.")
        return
    }

    // Use tabwriter for aligned columns
    w := tabwriter.NewWriter(os.Stdout, 0, 0, 2, ' ', 0)
    fmt.Fprintln(w, "ID\tTITLE\tPRIORITY\tSTATUS\tTAGS")
    fmt.Fprintln(w, "--\t-----\t--------\t------\t----")

    for _, task := range tasks {
        // Format tags
        tags := ""
        for i, tag := range task.Tags {
            if i > 0 {
                tags += ", "
            }
            tags += tag.Name
        }

        // Colorize status
        statusColor := getStatusColor(task.Status)

        fmt.Fprintf(w, "%d\t%s\t%s\t%s\t%s\n",
            task.ID,
            task.Title,
            task.Priority,
            statusColor.Sprint(task.Status),
            tags,
        )
    }

    w.Flush()
}

func getStatusColor(status models.Status) *color.Color {
    switch status {
    case models.StatusCompleted:
        return color.New(color.FgGreen)
    case models.StatusInProgress:
        return color.New(color.FgYellow)
    case models.StatusCancelled:
        return color.New(color.FgRed)
    default:
        return color.New(color.FgWhite)
    }
}

func parseStatus(s string) models.Status {
    switch s {
    case "pending":
        return models.StatusPending
    case "progress", "in-progress":
        return models.StatusInProgress
    case "completed", "done":
        return models.StatusCompleted
    case "cancelled":
        return models.StatusCancelled
    default:
        return models.StatusPending
    }
}
