package commands

import (
    "fmt"
    "strconv"

    "github.com/fatih/color"
    "github.com/spf13/cobra"
    "github.com/yourusername/taskmaster/internal/models"
)

func updateCmd() *cobra.Command {
    var title, description string
    var priority, status int

    cmd := &cobra.Command{
        Use:   "update [task ID]",
        Short: "Update a task",
        Args:  cobra.ExactArgs(1),
        Example: `  taskmaster update 42 --title "New title"
  taskmaster update 42 --priority 3
  taskmaster update 42 --status 1`,
        RunE: func(cmd *cobra.Command, args []string) error {
            id, err := strconv.ParseUint(args[0], 10, 32)
            if err != nil {
                return fmt.Errorf("invalid task ID: %s", args[0])
            }

            updates := make(map[string]interface{})

            if title != "" {
                updates["title"] = title
            }
            if description != "" {
                updates["description"] = description
            }
            if priority >= 0 {
                updates["priority"] = models.Priority(priority)
            }
            if status >= 0 {
                updates["status"] = models.Status(status)
            }

            if len(updates) == 0 {
                return fmt.Errorf("no updates specified")
            }

            if err := taskService.UpdateTask(uint(id), updates); err != nil {
                return fmt.Errorf("failed to update task: %w", err)
            }

            color.Green("✓ Task updated successfully!")
            return nil
        },
    }

    cmd.Flags().StringVarP(&title, "title", "t", "", "Task title")
    cmd.Flags().StringVarP(&description, "description", "d", "", "Task description")
    cmd.Flags().IntVarP(&priority, "priority", "p", -1, "Priority (0=Low, 1=Medium, 2=High, 3=Critical)")
    cmd.Flags().IntVarP(&status, "status", "s", -1, "Status (0=Pending, 1=InProgress, 2=Completed, 3=Cancelled)")

    return cmd
}
