package models

import (
    "time"
    "gorm.io/gorm"
)

// Task represents a task item
type Task struct {
    ID          uint           `gorm:"primarykey" json:"id"`
    Title       string         `gorm:"not null" json:"title"`
    Description string         `json:"description"`
    Priority    Priority       `gorm:"default:2" json:"priority"`  // Default: Medium
    Status      Status         `gorm:"default:0" json:"status"`     // Default: Pending
    DueDate     *time.Time     `json:"due_date,omitempty"`
    Tags        []Tag          `gorm:"many2many:task_tags;" json:"tags"`
    CreatedAt   time.Time      `json:"created_at"`
    UpdatedAt   time.Time      `json:"updated_at"`
    DeletedAt   gorm.DeletedAt `gorm:"index" json:"-"`  // Soft delete support
}

// Priority levels (0-3 for easy sorting)
type Priority int

const (
    PriorityLow Priority = iota      // 0
    PriorityMedium                    // 1
    PriorityHigh                      // 2
    PriorityCritical                  // 3
)

func (p Priority) String() string {
    return [...]string{"Low", "Medium", "High", "Critical"}[p]
}

// Task lifecycle states
type Status int

const (
    StatusPending Status = iota       // 0
    StatusInProgress                  // 1
    StatusCompleted                   // 2
    StatusCancelled                   // 3
)

func (s Status) String() string {
    return [...]string{"Pending", "In Progress", "Completed", "Cancelled"}[s]
}

// Tag for categorization (many-to-many with tasks)
type Tag struct {
    ID   uint   `gorm:"primarykey"`
    Name string `gorm:"unique;not null"`
}

// TaskFilter for query building
type TaskFilter struct {
    Status   *Status    // Filter by status (nil = all)
    Priority *Priority  // Filter by priority (nil = all)
    Tag      string     // Filter by tag name
    Search   string     // Search in title/description
}
