package repository

import (
    "errors"
    "github.com/yourusername/taskmaster/internal/models"
    "gorm.io/gorm"
)

var (
    ErrTaskNotFound = errors.New("task not found")
)

// TaskRepository defines data access operations
type TaskRepository interface {
    Create(task *models.Task) error
    GetByID(id uint) (*models.Task, error)
    List(filter models.TaskFilter) ([]*models.Task, error)
    Update(task *models.Task) error
    Delete(id uint) error
    GetStats() (*Stats, error)
}

// Stats for dashboard
type Stats struct {
    Total       int64
    Completed   int64
    Pending     int64
    InProgress  int64
    Cancelled   int64
}

// Implementation
type taskRepository struct {
    db *gorm.DB
}

func NewTaskRepository(db *gorm.DB) TaskRepository {
    return &taskRepository{db: db}
}

func (r *taskRepository) Create(task *models.Task) error {
    return r.db.Create(task).Error
}

func (r *taskRepository) GetByID(id uint) (*models.Task, error) {
    var task models.Task
    // Preload tags (solve N+1 query problem)
    err := r.db.Preload("Tags").First(&task, id).Error
    if errors.Is(err, gorm.ErrRecordNotFound) {
        return nil, ErrTaskNotFound
    }
    return &task, err
}

func (r *taskRepository) List(filter models.TaskFilter) ([]*models.Task, error) {
    var tasks []*models.Task
    query := r.db.Preload("Tags")

    // Dynamic query building based on filters
    if filter.Status != nil {
        query = query.Where("status = ?", *filter.Status)
    }

    if filter.Priority != nil {
        query = query.Where("priority = ?", *filter.Priority)
    }

    if filter.Tag != "" {
        // Join through the many-to-many relationship
        query = query.Joins("JOIN task_tags ON task_tags.task_id = tasks.id").
            Joins("JOIN tags ON tags.id = task_tags.tag_id").
            Where("tags.name = ?", filter.Tag)
    }

    if filter.Search != "" {
        // Full-text search in title and description
        searchPattern := "%" + filter.Search + "%"
        query = query.Where("title LIKE ? OR description LIKE ?",
            searchPattern, searchPattern)
    }

    // Sort by priority (high first) then by creation date
    err := query.Order("priority DESC, created_at DESC").Find(&tasks).Error
    return tasks, err
}

func (r *taskRepository) Update(task *models.Task) error {
    return r.db.Save(task).Error
}

func (r *taskRepository) Delete(id uint) error {
    result := r.db.Delete(&models.Task{}, id)
    if result.RowsAffected == 0 {
        return ErrTaskNotFound
    }
    return result.Error
}

func (r *taskRepository) GetStats() (*Stats, error) {
    var stats Stats

    r.db.Model(&models.Task{}).Count(&stats.Total)
    r.db.Model(&models.Task{}).Where("status = ?", models.StatusCompleted).Count(&stats.Completed)
    r.db.Model(&models.Task{}).Where("status = ?", models.StatusPending).Count(&stats.Pending)
    r.db.Model(&models.Task{}).Where("status = ?", models.StatusInProgress).Count(&stats.InProgress)
    r.db.Model(&models.Task{}).Where("status = ?", models.StatusCancelled).Count(&stats.Cancelled)

    return &stats, nil
}
