package service

import (
    "errors"
    "strings"
    "github.com/yourusername/taskmaster/internal/models"
    "github.com/yourusername/taskmaster/internal/repository"
)

var (
    ErrEmptyTitle = errors.New("task title cannot be empty")
)

// TaskService defines business operations
type TaskService interface {
    CreateTask(title, description string, priority models.Priority, tags []string) (*models.Task, error)
    GetTask(id uint) (*models.Task, error)
    ListTasks(filter models.TaskFilter) ([]*models.Task, error)
    CompleteTask(id uint) error
    UpdateTask(id uint, updates map[string]interface{}) error
    DeleteTask(id uint) error
    GetStatistics() (*repository.Stats, error)
}

type taskService struct {
    repo repository.TaskRepository
}

func NewTaskService(repo repository.TaskRepository) TaskService {
    return &taskService{repo: repo}
}

func (s *taskService) CreateTask(title, description string, priority models.Priority, tags []string) (*models.Task, error) {
    // Validation
    title = strings.TrimSpace(title)
    if title == "" {
        return nil, ErrEmptyTitle
    }

    // Create task
    task := &models.Task{
        Title:       title,
        Description: strings.TrimSpace(description),
        Priority:    priority,
        Status:      models.StatusPending,
    }

    // Convert tag names to Tag objects
    for _, tagName := range tags {
        tagName = strings.TrimSpace(tagName)
        if tagName != "" {
            task.Tags = append(task.Tags, models.Tag{Name: tagName})
        }
    }

    if err := s.repo.Create(task); err != nil {
        return nil, err
    }

    return task, nil
}

func (s *taskService) GetTask(id uint) (*models.Task, error) {
    return s.repo.GetByID(id)
}

func (s *taskService) ListTasks(filter models.TaskFilter) ([]*models.Task, error) {
    return s.repo.List(filter)
}

func (s *taskService) CompleteTask(id uint) error {
    task, err := s.repo.GetByID(id)
    if err != nil {
        return err
    }

    task.Status = models.StatusCompleted
    return s.repo.Update(task)
}

func (s *taskService) UpdateTask(id uint, updates map[string]interface{}) error {
    task, err := s.repo.GetByID(id)
    if err != nil {
        return err
    }

    // Apply updates safely
    if title, ok := updates["title"].(string); ok && title != "" {
        task.Title = strings.TrimSpace(title)
    }
    if desc, ok := updates["description"].(string); ok {
        task.Description = strings.TrimSpace(desc)
    }
    if priority, ok := updates["priority"].(models.Priority); ok {
        task.Priority = priority
    }
    if status, ok := updates["status"].(models.Status); ok {
        task.Status = status
    }

    return s.repo.Update(task)
}

func (s *taskService) DeleteTask(id uint) error {
    return s.repo.Delete(id)
}

func (s *taskService) GetStatistics() (*repository.Stats, error) {
    return s.repo.GetStats()
}
