package tests

import (
    "testing"

    "github.com/stretchr/testify/assert"
    "github.com/stretchr/testify/require"
    "github.com/yourusername/taskmaster/internal/models"
    "github.com/yourusername/taskmaster/internal/repository"
    "github.com/yourusername/taskmaster/internal/service"
    "gorm.io/driver/sqlite"
    "gorm.io/gorm"
)

func setupTestDB(t *testing.T) *gorm.DB {
    // Use in-memory database for fast tests
    db, err := gorm.Open(sqlite.Open(":memory:"), &gorm.Config{})
    require.NoError(t, err)

    err = db.AutoMigrate(&models.Task{}, &models.Tag{})
    require.NoError(t, err)

    return db
}

func TestTaskLifecycle(t *testing.T) {
    db := setupTestDB(t)
    repo := repository.NewTaskRepository(db)
    svc := service.NewTaskService(repo)

    // Create task
    task, err := svc.CreateTask("Test Task", "Description", models.PriorityHigh, []string{"test"})
    require.NoError(t, err)
    assert.NotZero(t, task.ID)
    assert.Equal(t, "Test Task", task.Title)
    assert.Equal(t, models.StatusPending, task.Status)

    // Get task
    retrieved, err := svc.GetTask(task.ID)
    require.NoError(t, err)
    assert.Equal(t, task.ID, retrieved.ID)

    // Complete task
    err = svc.CompleteTask(task.ID)
    require.NoError(t, err)

    completed, err := svc.GetTask(task.ID)
    require.NoError(t, err)
    assert.Equal(t, models.StatusCompleted, completed.Status)

    // Delete task
    err = svc.DeleteTask(task.ID)
    require.NoError(t, err)

    _, err = svc.GetTask(task.ID)
    assert.ErrorIs(t, err, repository.ErrTaskNotFound)
}

func TestTaskFiltering(t *testing.T) {
    db := setupTestDB(t)
    repo := repository.NewTaskRepository(db)
    svc := service.NewTaskService(repo)

    // Create test tasks
    svc.CreateTask("Task 1", "", models.PriorityHigh, []string{"urgent"})
    svc.CreateTask("Task 2", "", models.PriorityLow, []string{"later"})
    svc.CreateTask("Task 3", "", models.PriorityHigh, []string{"urgent"})

    // Filter by priority
    highPriority := models.PriorityHigh
    tasks, err := svc.ListTasks(models.TaskFilter{Priority: &highPriority})
    require.NoError(t, err)
    assert.Len(t, tasks, 2)

    // Filter by tag
    tasks, err = svc.ListTasks(models.TaskFilter{Tag: "urgent"})
    require.NoError(t, err)
    assert.Len(t, tasks, 2)

    // Search
    tasks, err = svc.ListTasks(models.TaskFilter{Search: "Task 2"})
    require.NoError(t, err)
    assert.Len(t, tasks, 1)
    assert.Equal(t, "Task 2", tasks[0].Title)
}
