# Architecture Documentation

Comprehensive architecture guide for the Real-time Chat Server.

## System Design

### High-Level Architecture

```
┌─────────────────────────────────────────────────────────────────┐
│                         Load Balancer                            │
│                    (nginx, AWS ALB, etc.)                        │
└───────────────┬─────────────────────┬───────────────────────────┘
                │                     │
    ┌───────────▼──────────┐  ┌──────▼──────────────┐
    │  Chat Server 1       │  │  Chat Server 2      │
    │  ┌────────────────┐  │  │  ┌────────────────┐ │
    │  │  HTTP Router   │  │  │  │  HTTP Router   │ │
    │  │    (Chi)       │  │  │  │    (Chi)       │ │
    │  └────────┬───────┘  │  │  └────────┬───────┘ │
    │           │          │  │           │         │
    │  ┌────────▼────────┐ │  │  ┌────────▼───────┐ │
    │  │ WebSocket       │ │  │  │ WebSocket      │ │
    │  │ Handler         │ │  │  │ Handler        │ │
    │  └────────┬────────┘ │  │  └────────┬───────┘ │
    │           │          │  │           │         │
    │  ┌────────▼────────┐ │  │  ┌────────▼───────┐ │
    │  │      Hub        │ │  │  │      Hub       │ │
    │  │  (Connections)  │ │  │  │  (Connections) │ │
    │  └────────┬────────┘ │  │  └────────┬───────┘ │
    └───────────┼──────────┘  └───────────┼─────────┘
                │                         │
                │   ┌─────────────────┐   │
                └───► Redis Pub/Sub   ◄───┘
                    │   (Messaging)   │
                    └─────────────────┘
                            │
                    ┌───────▼─────────┐
                    │    MongoDB      │
                    │  (Persistence)  │
                    └─────────────────┘
```

## Component Architecture

### 1. HTTP Router (Chi)

**File**: `cmd/server/main.go`

**Responsibilities**:
- Route incoming HTTP requests
- Configure middleware stack
- Register handlers
- Manage graceful shutdown

**Key Features**:
- CORS support
- Request logging
- Panic recovery
- Request/response handling

**Middleware Stack**:
```
Request
  ↓
Logger (logs all requests)
  ↓
Recoverer (catches panics)
  ↓
CORS (allows cross-origin requests)
  ↓
Handler (processes request)
  ↓
Response
```

### 2. WebSocket Handler

**File**: `internal/handlers/websocket.go`

**Responsibilities**:
- HTTP to WebSocket protocol upgrade
- Connection initialization
- Client registration
- Message handling coordination

**Key Methods**:

#### ServeWS
```
HTTP Upgrade → WebSocket Connection → Client Registration → Pump Initialization
```

#### Read Pump
```
Client WebSocket Input → Unmarshal JSON → Database Save → Hub Broadcast
```

#### Write Pump
```
Hub Channel → Format Response → Write to WebSocket → Heartbeat Management
```

**Connection Lifecycle**:
```
1. Client requests ws://host/ws?user_id=X&room_id=Y
2. Handler upgrades HTTP to WebSocket
3. Creates Client object
4. Registers with Hub
5. Starts read and write pumps
6. Pumps continue until connection closes
7. Handler unregisters from Hub
```

### 3. Hub (Message Broker)

**File**: `internal/hub/hub.go`

**Responsibilities**:
- Central message routing
- Connection management
- Room-based broadcasting
- Redis pub/sub coordination

**Data Structures**:

```go
type Hub struct {
    clients    map[string]*Client                // Global client registry
    rooms      map[string]map[string]*Client     // Room-based clients
    broadcast  chan *models.Message              // Incoming messages
    register   chan *Client                      // New connections
    unregister chan *Client                      // Disconnections
    pubsub     *pubsub.RedisPubSub               // Cross-server messaging
    mu         sync.RWMutex                      // Thread safety
}
```

**Event Loop**:
```
Main goroutine continuously selects from three channels:

1. register channel
   - Adds client to local maps
   - Initializes room if needed
   - Logs connection

2. unregister channel
   - Removes client from maps
   - Closes client channels
   - Notifies other servers via Redis

3. broadcast channel
   - Receives message from handlers
   - Publishes to Redis
   - Broadcasts to local clients
```

**Message Broadcasting Flow**:
```
Message received from client
    ↓
Handler's readPump receives
    ↓
Save to MongoDB
    ↓
Send to Hub.broadcast channel
    ↓
Hub publishes to Redis
    ↓
All servers receive via Redis subscription
    ↓
Hub broadcasts to room clients locally
    ↓
Write pump sends to each connected client
```

### 4. Redis Pub/Sub

**File**: `internal/pubsub/redis.go`

**Responsibilities**:
- Cross-server message distribution
- Horizontal scaling enablement
- Stateless server operation

**Implementation**:
```
Server 1: Publishes "chat" channel message
    ↓
Redis: Broadcasts to all subscribers
    ↓
Server 1: Receives and broadcasts locally
Server 2: Receives and broadcasts locally
Server 3: Receives and broadcasts locally
```

**Benefits**:
- Servers don't maintain knowledge of each other
- Messages delivered to all servers automatically
- Easy to add/remove servers
- Automatic failover when servers go down

### 5. MongoDB Repository

**File**: `internal/repository/message_repository.go`

**Responsibilities**:
- Persist all chat messages
- Retrieve message history
- Query messages by criteria
- Handle database operations

**Collections**:
- `messages`: All chat messages
  - Indexes on: room_id, user_id, timestamp

**Operations**:
- `Create`: Insert new message
- `GetByRoom`: Retrieve messages for a room
- `Delete`: Remove message (admin/moderation)

**Database Schema**:
```
Message {
  _id:       ObjectID
  room_id:   ObjectID
  user_id:   ObjectID
  username:  string
  content:   string
  type:      enum(text, join, leave, system)
  timestamp: datetime
}
```

### 6. Data Models

**File**: `internal/models/`

#### User Model
```go
type User struct {
    ID        ObjectID  // MongoDB ID
    Username  string    // Display name
    Email     string    // Email
    Password  string    // Hashed (not exposed)
    Avatar    string    // Profile picture URL
    Status    UserStatus // online, offline, away
    CreatedAt time.Time
    UpdatedAt time.Time
}
```

#### Room Model
```go
type Room struct {
    ID          ObjectID   // MongoDB ID
    Name        string     // Display name
    Description string     // Room description
    CreatorID   ObjectID   // Creator user ID
    Members     []ObjectID // Member IDs
    IsPrivate   bool       // Public vs private
    CreatedAt   time.Time
    UpdatedAt   time.Time
}
```

#### Message Model
```go
type Message struct {
    ID        ObjectID   // MongoDB ID
    RoomID    ObjectID   // Room ID
    UserID    ObjectID   // User ID
    Username  string     // Cached username
    Content   string     // Message text
    Type      MessageType // text, join, leave, system
    Timestamp time.Time
}

type ClientMessage struct {
    Type    string // Message type
    RoomID  string // Target room
    Content string // Message content
}
```

## Data Flow

### Message Sending Flow

```
1. Client sends message via WebSocket
   └─ Format: {"type":"message","room_id":"...","content":"..."}

2. Handler's readPump receives
   └─ Reads from WebSocket connection

3. Parse and validate
   └─ JSON unmarshaling
   └─ Input validation

4. Create Message object
   └─ Set UserID, RoomID, Timestamp
   └─ Type = MessageTypeText

5. Save to MongoDB
   └─ Insert operation with context timeout
   └─ Generate new ObjectID

6. Send to Hub.broadcast channel
   └─ Non-blocking send with buffer

7. Hub.Run() receives broadcast
   └─ Marshal to JSON
   └─ Publish to Redis "chat" channel

8. All servers receive via Redis subscription
   └─ Each server's Hub receives

9. Hub broadcasts to room clients
   └─ Iterate through room's clients
   └─ Send to each client's Send channel

10. Write pump receives from Send channel
    └─ Set write deadline
    └─ Write to WebSocket

11. Client receives message
    └─ Displayed in UI
```

### Connection Lifecycle

```
1. Client connects
   ├─ HTTP request to /ws endpoint
   ├─ Handler checks origin
   ├─ Upgrade to WebSocket protocol
   └─ Send initial headers

2. Handler initializes
   ├─ Create Client object with unique ID
   ├─ Set up 256-byte buffered Send channel
   ├─ Extract user_id and room_id from query
   └─ Create client

3. Register with Hub
   ├─ Send to Hub.register channel
   ├─ Hub adds to global clients map
   ├─ Hub adds to room's client map
   └─ Log connection

4. Start pumps (concurrent)
   ├─ readPump (goroutine 1)
   │  └─ Listen for client messages
   └─ writePump (goroutine 2)
      └─ Send server messages

5. Active communication
   ├─ Messages flow through pumps
   ├─ Heartbeat pings every 54 seconds
   └─ Pongs reset read deadline

6. Disconnection (when connection closes)
   ├─ readPump breaks from loop
   ├─ Defers unregister
   ├─ Handler unregisters from Hub
   ├─ Hub removes from maps
   └─ Broadcast unregister to other servers

7. Cleanup
   ├─ Close WebSocket connection
   ├─ Close Send channel
   ├─ All related goroutines terminate
   └─ Resources freed
```

## Concurrency Model

### Goroutine Management

```
Main Goroutine
    ├─ HTTP Server (blocks)
    ├─ Hub.Run() (main event loop)
    │   └─ Runs continuously until app shutdown
    │
    └─ For each WebSocket connection:
        ├─ readPump() goroutine
        │   └─ Blocks reading from WebSocket
        ├─ writePump() goroutine
        │   └─ Blocks on select{} for messages or ticks
        └─ Redis subscription (in Hub setup)
            └─ Blocks listening for Redis messages
```

### Synchronization

**Channels Used**:
- `Hub.register` - Unbuffered, used to register clients
- `Hub.unregister` - Unbuffered, used to unregister clients
- `Hub.broadcast` - 256-byte buffered, for messages to broadcast
- `Client.Send` - 256-byte buffered, for messages to client

**Mutexes Used**:
- `Hub.mu` - RWMutex for thread-safe access to maps
  - RLock for reading (broadcasting)
  - Lock for writing (register/unregister)

**Pattern**:
```go
// Writing to shared data
h.mu.Lock()
h.clients[id] = client
h.mu.Unlock()

// Reading for broadcast
h.mu.RLock()
clients := h.rooms[roomID]
h.mu.RUnlock()
// Now safely read clients without lock
```

## Performance Characteristics

### Memory Usage

**Per Connection**:
- Client struct: ~500 bytes
- Send channel buffer: 256 * ~8KB = ~2MB average
- WebSocket connection buffers: 1KB read + 1KB write
- **Total per connection**: ~2.5MB average

**Scalability**: With 10,000 connections = ~25GB memory

**Optimization**: 
- Reduce channel buffer sizes for memory efficiency
- Implement connection limits based on available resources

### Message Latency

**Path**: Client send → readPump → MongoDB → Hub → Redis → writePump → Client receive

**Timing**:
- WebSocket read: 1-5ms
- JSON parsing: 0.1-0.5ms
- MongoDB insert: 1-10ms (depends on load)
- Redis publish: 0.5-2ms
- Local hub broadcast: 0.5-5ms (per connection)
- WebSocket write: 1-5ms

**Total**: ~5-35ms end-to-end (varies with load)

### Throughput

**Messages per second**:
- Single server: 1,000-5,000 msg/s (depends on message size)
- Multiple servers: Scales with Redis bandwidth
- MongoDB write throughput: 5,000-10,000 ops/s (depends on hardware)

**Bottlenecks**:
- MongoDB write latency (usually largest)
- Network bandwidth to/from Redis
- Client connection speed

## Scaling Strategies

### Horizontal Scaling

**Add more servers**:
1. Deploy new server instance
2. Points to same MongoDB and Redis
3. Automatically joins the distributed system
4. No configuration changes needed

**Load Balancing**:
- Use sticky sessions (optional, depending on requirements)
- Or stateless design (current implementation)

### Vertical Scaling

**Optimize single server**:
- Increase channel buffer sizes (use more memory)
- Reduce read/write timeouts
- Profile and optimize hot paths
- Use pprof for analysis

### Database Scaling

**MongoDB**:
- Use replication for HA
- Sharding for data distribution
- Indexes on room_id and user_id

**Redis**:
- Use cluster for distribution
- Sentinel for failover
- Consider Redis Streams for persistent pub/sub

## Security Architecture

### Authentication

- JWT tokens for user identification
- Tokens passed via query parameter (in real scenario, use headers)
- Token validation on connection
- Token expiration handling

### Input Validation

- Message content length limits (5,000 chars max)
- Empty content rejection
- Type validation
- RoomID and UserID validation

### Transport Security

- HTTPS/WSS for production (via reverse proxy)
- Origin validation for CORS
- No sensitive data in logs
- Input sanitization

## Error Handling

### Connection Errors

- Read/write deadlines prevent hanging connections
- Ping/pong detects dead connections
- Graceful closure on error
- Automatic cleanup on disconnect

### Message Errors

- JSON parsing failures logged
- Invalid messages ignored
- Database errors logged but don't crash server
- Client notified of errors via message types

### System Errors

- MongoDB connection errors caught at startup
- Redis connection errors caught at startup
- Middleware recovers from panics
- Graceful shutdown on signals

## Testing Architecture

### Unit Tests

Test individual components in isolation:
- Hub registration/unregistration
- Message routing
- Repository operations
- Authentication

### Integration Tests

Test full message flow:
- WebSocket connection → message → database → broadcast
- Multiple connections in same room
- Multiple connections in different rooms

### Load Tests

Test system under stress:
- k6 load testing
- Connection scaling
- Message throughput
- Memory usage under load

---

For more details, see [README.md](README.md)
