# Contributing to Real-time Chat Server

Thank you for your interest in contributing! This document provides guidelines for making contributions.

## Code of Conduct

- Be respectful and constructive
- Provide clear, detailed information in issues and PRs
- Test your changes thoroughly
- Follow the existing code style and patterns

## Getting Started

1. Fork the repository
2. Clone your fork: `git clone https://github.com/yourusername/chatserver.git`
3. Create a feature branch: `git checkout -b feature/your-feature-name`
4. Install dependencies: `go mod download`

## Development Setup

### Prerequisites
- Go 1.24+
- Docker and Docker Compose
- MongoDB (local or Docker)
- Redis (local or Docker)

### Running Locally

```bash
# Start services
docker-compose up -d

# Run tests
make test

# Build and run
make build
make run
```

## Code Standards

### Go Code Style

- Run `make fmt` to format code
- Run `make vet` to check for issues
- Run `make lint` for linting (if golangci-lint installed)
- Maintain test coverage above 70%

### Naming Conventions

- Package names: lowercase, concise
- Interface names: use `er` suffix (e.g., `Reader`, `Writer`)
- Unexported identifiers: start with lowercase
- Constants: `UPPER_CASE`

### Comments

- Exported functions/types must have comments
- Comments should be clear and concise
- Use proper English grammar

## Testing

### Running Tests

```bash
# All tests
make test

# Specific package
go test ./internal/hub/...

# With verbose output
go test -v ./...

# With race detector
go test -race ./...

# Coverage
make test-coverage
```

### Writing Tests

- Use `*testing.T` parameter
- Follow `TestFunctionName` naming
- Use table-driven tests for multiple cases
- Mock external dependencies
- Test both success and error cases

Example:
```go
func TestHubRegister(t *testing.T) {
    tests := []struct {
        name    string
        client  *Client
        wantErr bool
    }{
        {
            name: "successful registration",
            client: &Client{ID: "test-1"},
            wantErr: false,
        },
    }

    for _, tt := range tests {
        t.Run(tt.name, func(t *testing.T) {
            // Test code
        })
    }
}
```

## Git Workflow

### Commit Messages

Follow this format:
```
type(scope): subject

body

footer
```

Types:
- `feat`: New feature
- `fix`: Bug fix
- `docs`: Documentation
- `style`: Code style changes
- `refactor`: Code refactoring
- `perf`: Performance improvement
- `test`: Test changes

Example:
```
feat(hub): add client broadcasting

Implement message broadcasting to all clients in a room
using the Hub pattern with goroutines.

Closes #42
```

### Pull Request Process

1. Update dependencies: `go mod tidy`
2. Run tests: `make test`
3. Run code checks: `make check`
4. Write clear PR description
5. Link related issues
6. Wait for review
7. Address feedback
8. Merge when approved

## Pull Request Template

```markdown
## Description
Brief description of changes

## Type of Change
- [ ] Bug fix
- [ ] New feature
- [ ] Breaking change
- [ ] Documentation update

## Testing
- [ ] Unit tests added
- [ ] Integration tests added
- [ ] Manual testing completed

## Checklist
- [ ] Code follows style guidelines
- [ ] Tests pass locally
- [ ] Documentation updated
- [ ] No new warnings
```

## Reporting Bugs

### Bug Report Template

```markdown
## Description
Clear description of the bug

## Steps to Reproduce
1. Step 1
2. Step 2
3. Expected behavior
4. Actual behavior

## Environment
- Go version: 
- OS: 
- Docker version:
- MongoDB version:
- Redis version:

## Logs/Error Messages
Include relevant logs or error output
```

## Suggesting Features

### Feature Request Template

```markdown
## Description
Clear description of the desired feature

## Motivation
Why is this needed?

## Proposed Solution
How should it work?

## Alternatives
Other approaches considered?
```

## Documentation

- Update README.md for user-facing changes
- Add comments to complex code
- Update API docs for endpoint changes
- Include examples where helpful

## Performance

Consider performance implications:
- Profile changes: `make bench`
- Monitor memory usage
- Check for goroutine leaks
- Use buffered channels appropriately

## Security

- Don't commit secrets or sensitive data
- Validate all inputs
- Use secure defaults
- Document security implications
- Check dependencies for vulnerabilities

## Building and Releasing

### Local Build

```bash
make docker-build
```

### Pushing Changes

```bash
git push origin feature/your-feature-name
```

Then open a pull request on GitHub.

## Questions?

- Check existing issues and discussions
- Review the main README.md
- Check the troubleshooting section
- Open an issue with the question

## License

By contributing, you agree that your contributions will be licensed under the MIT License.

---

Thank you for contributing!
