# Quick Start Guide

Get the Real-time Chat Server running in 5 minutes!

## Prerequisites

- Docker and Docker Compose installed
- Git (optional, if cloning)

## Option 1: Docker Compose (Recommended)

### 1. Start Services

```bash
docker-compose up -d
```

This starts:
- Go backend on `http://localhost:8080`
- MongoDB on `http://localhost:27017`
- Redis on `http://localhost:6379`

### 2. Verify Services

```bash
# Check running containers
docker-compose ps

# View logs
docker-compose logs -f app
```

### 3. Test WebSocket Connection

Open a new terminal and test the WebSocket:

```bash
# Install websocat if you don't have it
# On macOS: brew install websocat
# On Linux: cargo install websocat

# Connect to WebSocket
websocat "ws://localhost:8080/ws?user_id=user1&room_id=room1"

# In the WebSocket connection, send a message:
{"type":"message","room_id":"room1","content":"Hello, World!"}
```

Or use a browser console:

```javascript
const ws = new WebSocket('ws://localhost:8080/ws?user_id=user1&room_id=room1');
ws.onopen = () => {
  ws.send(JSON.stringify({
    type: 'message',
    room_id: 'room1',
    content: 'Hello from browser!'
  }));
};
ws.onmessage = (event) => {
  console.log('Received:', event.data);
};
```

### 4. Check Health

```bash
curl http://localhost:8080/health
# Should respond with: OK
```

### 5. Verify MongoDB

```bash
docker-compose exec mongo mongosh

# Inside mongosh:
use chatserver
db.messages.find().pretty()
exit
```

### 6. Verify Redis

```bash
docker-compose exec redis redis-cli

# Inside redis-cli:
PING
# Should respond with: PONG

MONITOR
# Watch for pub/sub messages

exit
```

## Option 2: Local Development

### 1. Start Database Services

```bash
# Terminal 1: MongoDB
docker run -d -p 27017:27017 mongo:7

# Terminal 2: Redis
docker run -d -p 6379:6379 redis:7-alpine
```

### 2. Run the Application

```bash
# Terminal 3: Chat Server
go run cmd/server/main.go
```

Server starts on `http://localhost:8080`

## Option 3: Build Binary

```bash
# Build
make build

# Run
MONGO_URI=mongodb://localhost:27017 \
REDIS_ADDR=localhost:6379 \
PORT=8080 \
./bin/server
```

## Using the React Frontend

### Prerequisites

- Node.js 16+ installed
- npm or yarn

### Setup

```bash
# Install dependencies
npm install

# Start development server
npm start
```

Frontend runs on `http://localhost:3000`

### Update WebSocket URL

If your backend is not on localhost:8080, update the URL in `web/src/components/Chat.jsx`:

```javascript
ws.current = new WebSocket(
  `ws://YOUR_SERVER_ADDRESS:PORT/ws?room_id=${roomId}&user_id=${userId}`
);
```

## Common Tasks

### View Logs

```bash
# Docker Compose
docker-compose logs -f app

# Or follow specific service
docker-compose logs -f mongo
docker-compose logs -f redis
```

### Stop Services

```bash
# Pause services (keeps data)
docker-compose stop

# Remove containers but keep volumes
docker-compose down

# Remove everything including data
docker-compose down -v
```

### Rebuild

```bash
# Rebuild container
docker-compose build

# Start fresh
docker-compose down -v
docker-compose up -d
```

### Run Tests

```bash
make test

# With coverage
make test-coverage
```

## Troubleshooting

### "Connection refused" error

Check if services are running:
```bash
docker-compose ps
```

All three services should show "Up".

### MongoDB connection error

```bash
# Check MongoDB logs
docker-compose logs mongo

# Connect manually
docker-compose exec mongo mongosh
```

### Redis connection error

```bash
# Check Redis logs
docker-compose logs redis

# Test connection
docker-compose exec redis redis-cli ping
```

### Port already in use

If port 8080, 27017, or 6379 are already in use, modify `docker-compose.yml`:

```yaml
services:
  app:
    ports:
      - "8081:8080"  # Changed from 8080:8080
```

## Next Steps

1. Read the [main README.md](README.md) for detailed documentation
2. Check [CONTRIBUTING.md](CONTRIBUTING.md) to contribute
3. Review the [Troubleshooting](README.md#troubleshooting) section
4. Explore the [API Reference](README.md#api-reference)

## Support

For issues or questions:
- Check existing issues on GitHub
- Review the main README.md
- Open a new issue with details
