# Real-time Chat Server - Complete Implementation Guide

A production-ready, horizontally scalable real-time chat server built with Go, WebSockets, Redis, and MongoDB. This comprehensive guide covers architecture, implementation details, and deployment strategies.

## Table of Contents

- [Overview](#overview)
- [Features](#features)
- [Architecture Deep Dive](#architecture-deep-dive)
- [Technology Stack](#technology-stack)
- [Project Structure](#project-structure)
- [Quick Start](#quick-start)
- [Implementation Guide](#implementation-guide)
- [Testing Strategy](#testing-strategy)
- [Deployment Guide](#deployment-guide)
- [Performance Optimization](#performance-optimization)
- [Security Considerations](#security-considerations)
- [Troubleshooting](#troubleshooting)

---

## Overview

This real-time chat server demonstrates production-ready patterns for building scalable WebSocket applications in Go. It handles thousands of concurrent connections, supports horizontal scaling through Redis pub/sub, and provides persistent message storage with MongoDB.

### Key Capabilities

- **Real-time bi-directional communication** using WebSocket protocol
- **Horizontal scalability** via Redis pub/sub messaging
- **Message persistence** with MongoDB for chat history
- **Room-based chat** supporting multiple isolated chat rooms
- **JWT authentication** for secure user verification
- **Connection health monitoring** with automatic heartbeats
- **Graceful shutdown** ensuring no message loss during deployment

---

## Features

### Core Features

1. **WebSocket Communication**
   - Full-duplex communication over WebSocket
   - Support for multiple concurrent connections
   - Automatic reconnection handling
   - Message queuing during network issues

2. **Horizontal Scaling**
   - Redis pub/sub for cross-server messaging
   - Support for multiple server instances
   - Load balancer compatible
   - Shared message state across servers

3. **Message Persistence**
   - MongoDB storage for all messages
   - Message history retrieval
   - Query messages by room and timestamp
   - Efficient indexing for fast queries

4. **User Authentication**
   - JWT-based authentication
   - Secure token generation and validation
   - User session management
   - Token expiration handling

5. **Room Management**
   - Multiple chat rooms
   - Private and public room support
   - Room member tracking
   - Real-time room statistics

6. **Connection Management**
   - Ping/pong heartbeats every 54 seconds
   - Automatic dead connection cleanup
   - Read/write deadlines for timeout handling
   - Graceful disconnection

---

## Architecture Deep Dive

### System Architecture

```
┌─────────────────────────────────────────────────────────────────┐
│                         Load Balancer                            │
│                    (nginx, AWS ALB, etc.)                        │
└───────────────┬─────────────────────┬───────────────────────────┘
                │                     │
    ┌───────────▼──────────┐  ┌──────▼──────────────┐
    │  Chat Server 1       │  │  Chat Server 2      │
    │  ┌────────────────┐  │  │  ┌────────────────┐ │
    │  │  HTTP Router   │  │  │  │  HTTP Router   │ │
    │  │    (Chi)       │  │  │  │    (Chi)       │ │
    │  └────────┬───────┘  │  │  └────────┬───────┘ │
    │           │          │  │           │         │
    │  ┌────────▼────────┐ │  │  ┌────────▼───────┐ │
    │  │ WebSocket       │ │  │  │ WebSocket      │ │
    │  │ Handler         │ │  │  │ Handler        │ │
    │  └────────┬────────┘ │  │  └────────┬───────┘ │
    │           │          │  │           │         │
    │  ┌────────▼────────┐ │  │  ┌────────▼───────┐ │
    │  │      Hub        │ │  │  │      Hub       │ │
    │  │  (Connections)  │ │  │  │  (Connections) │ │
    │  └────────┬────────┘ │  │  └────────┬───────┘ │
    └───────────┼──────────┘  └───────────┼─────────┘
                │                         │
                │   ┌─────────────────┐   │
                └───► Redis Pub/Sub   ◄───┘
                    │   (Messaging)   │
                    └─────────────────┘
                            │
                    ┌───────▼─────────┐
                    │    MongoDB      │
                    │  (Persistence)  │
                    └─────────────────┘
```

### Real-time Messaging Architecture

The chat server uses a **Hub pattern** to manage WebSocket connections and message distribution:

1. **Client Connection Flow**
   ```
   Client → HTTP Upgrade → WebSocket Handler → Hub Registration → Ready
   ```

2. **Message Broadcasting Flow**
   ```
   Client Message → WebSocket Handler → Hub Broadcast Channel
        ↓
   Save to MongoDB
        ↓
   Publish to Redis Channel
        ↓
   All Server Instances Subscribe → Hub Receives Message
        ↓
   Broadcast to Room Clients → WebSocket Write
   ```

3. **Connection Lifecycle**
   ```
   Connect → Register → Active (Read/Write Pumps) → Disconnect → Cleanup
   ```

### Component Interaction

#### Hub (Central Message Broker)

The Hub is the heart of the real-time messaging system:

```go
type Hub struct {
    clients    map[string]*Client           // All connected clients
    rooms      map[string]map[string]*Client // Clients organized by room
    broadcast  chan *models.Message          // Broadcast channel
    register   chan *Client                  // New client registration
    unregister chan *Client                  // Client disconnection
    pubsub     *pubsub.RedisPubSub          // Redis pub/sub
    mu         sync.RWMutex                  // Thread-safe access
}
```

**Key responsibilities:**
- Manage all active WebSocket connections
- Route messages to appropriate rooms
- Handle client registration/unregistration
- Coordinate with Redis for multi-server messaging
- Thread-safe concurrent access to connection maps

#### WebSocket Handler (Connection Manager)

Manages individual WebSocket connections with two goroutines:

1. **Read Pump** (Client → Server)
   - Reads incoming messages from WebSocket
   - Parses and validates message format
   - Saves messages to MongoDB
   - Broadcasts to Hub for distribution
   - Handles ping/pong for connection health

2. **Write Pump** (Server → Client)
   - Sends messages from Hub to client
   - Manages write deadlines
   - Sends periodic ping messages
   - Handles graceful connection closure

#### Redis Pub/Sub (Horizontal Scaling)

Enables multiple server instances to share messages:

```
Server 1 publishes message → Redis Channel → Server 2 subscribes
                                          → Server 3 subscribes
                                          → Server N subscribes
```

**Benefits:**
- Stateless server instances
- Easy horizontal scaling
- No server affinity required
- Seamless failover

#### MongoDB (Message Persistence)

Stores all chat messages for:
- Chat history retrieval
- Analytics and reporting
- Compliance and auditing
- Message search functionality

---

## Technology Stack

### Core Technologies

| Technology | Version | Purpose |
|------------|---------|---------|
| Go | 1.24+ | Backend language |
| gorilla/websocket | 1.5.1 | WebSocket protocol implementation |
| MongoDB | 7.0+ | Message persistence and storage |
| Redis | 7.0+ | Pub/sub for horizontal scaling |
| Chi | 5.0.12 | HTTP router and middleware |
| JWT | 5.2.1 | Authentication tokens |
| Docker | Latest | Containerization |
| Docker Compose | Latest | Multi-container orchestration |

### Go Dependencies

```go
require (
    github.com/go-chi/chi/v5 v5.0.12         // HTTP router
    github.com/go-chi/cors v1.2.1             // CORS middleware
    github.com/golang-jwt/jwt/v5 v5.2.1       // JWT authentication
    github.com/google/uuid v1.6.0             // UUID generation
    github.com/gorilla/websocket v1.5.1       // WebSocket support
    github.com/redis/go-redis/v9 v9.5.1       // Redis client
    go.mongodb.org/mongo-driver v1.15.0       // MongoDB driver
)
```

### Why These Technologies?

**Go**:
- Excellent concurrency support with goroutines
- Low memory footprint
- Fast compilation and execution
- Built-in HTTP/WebSocket support

**gorilla/websocket**:
- Production-tested WebSocket library
- RFC 6455 compliant
- Connection upgrading
- Ping/pong support

**MongoDB**:
- Flexible document schema
- Horizontal scaling with sharding
- Rich query capabilities
- Time-series data support

**Redis**:
- In-memory pub/sub for low latency
- Simple and fast
- Persistent connections
- Cluster support for HA

---

## Project Structure

```
chatserver-solution/
├── cmd/
│   └── server/
│       └── main.go                    # Application entry point
├── internal/
│   ├── auth/
│   │   └── jwt.go                     # JWT token generation/validation
│   ├── handlers/
│   │   └── websocket.go               # WebSocket connection handling
│   ├── hub/
│   │   └── hub.go                     # Connection hub and broadcasting
│   ├── models/
│   │   ├── message.go                 # Message data structures
│   │   ├── room.go                    # Room data structures
│   │   └── user.go                    # User data structures
│   ├── pubsub/
│   │   └── redis.go                   # Redis pub/sub implementation
│   └── repository/
│       └── message_repository.go      # MongoDB message operations
├── docker-compose.yml                 # Multi-container orchestration
├── Dockerfile                         # Container build instructions
├── go.mod                             # Go module dependencies
├── go.sum                             # Dependency checksums
└── README.md                          # This file
```

### File-by-File Breakdown

#### `/cmd/server/main.go` (107 lines)

**Purpose**: Application entry point and server initialization

**Key components**:
- MongoDB connection setup
- Redis connection initialization
- Hub creation and startup
- HTTP router configuration
- Middleware setup (CORS, logging, recovery)
- Graceful shutdown handling

**Code walkthrough**:

```go
func main() {
    // 1. MongoDB Connection
    // Creates connection to MongoDB with 10-second timeout
    // Uses environment variable MONGO_URI or defaults to localhost
    mongoURI := getEnv("MONGO_URI", "mongodb://localhost:27017")
    client, err := mongo.Connect(ctx, options.Client().ApplyURI(mongoURI))

    // 2. Redis Connection
    // Initializes Redis pub/sub for cross-server messaging
    redisAddr := getEnv("REDIS_ADDR", "localhost:6379")
    ps := pubsub.New(redisAddr)

    // 3. Hub Initialization
    // Creates the central hub and starts it in a goroutine
    h := hub.New(ps)
    go h.Run()

    // 4. Repository Setup
    // Creates message repository for database operations
    msgRepo := repository.NewMessageRepository(db)

    // 5. Handler Creation
    // Sets up WebSocket handler with hub and repository
    wsHandler := handlers.NewWebSocketHandler(h, msgRepo)

    // 6. Router Configuration
    // Sets up Chi router with middleware
    r := chi.NewRouter()
    r.Use(middleware.Logger)      // Request logging
    r.Use(middleware.Recoverer)   // Panic recovery
    r.Use(cors.Handler(...))      // CORS configuration

    // 7. Route Registration
    r.Get("/ws", wsHandler.ServeWS)      // WebSocket endpoint
    r.Get("/health", healthCheckHandler)  // Health check

    // 8. Server Startup
    // Starts HTTP server on specified port
    srv := &http.Server{Addr: ":8080", Handler: r}

    // 9. Graceful Shutdown
    // Listens for SIGINT/SIGTERM and shuts down gracefully
    quit := make(chan os.Signal, 1)
    signal.Notify(quit, syscall.SIGINT, syscall.SIGTERM)
    <-quit
    srv.Shutdown(ctx)
}
```

**Environment variables**:
- `MONGO_URI`: MongoDB connection string
- `REDIS_ADDR`: Redis server address
- `PORT`: HTTP server port

---

#### `/internal/models/user.go` (26 lines)

**Purpose**: User data structure and status constants

**Data structure**:

```go
type User struct {
    ID        primitive.ObjectID  // MongoDB unique identifier
    Username  string              // User's display name
    Email     string              // User's email address
    Password  string              // Hashed password (not exposed in JSON)
    Avatar    string              // Avatar URL or path
    Status    UserStatus          // Current online status
    CreatedAt time.Time           // Account creation timestamp
    UpdatedAt time.Time           // Last update timestamp
}

type UserStatus string

const (
    StatusOnline  UserStatus = "online"   // User is connected
    StatusOffline UserStatus = "offline"  // User is disconnected
    StatusAway    UserStatus = "away"     // User is idle
)
```

**Field explanations**:
- `ID`: MongoDB ObjectID for unique user identification
- `Username`: Display name shown in chat
- `Email`: Used for authentication and notifications
- `Password`: Stored as bcrypt hash, never sent to client (json:"-")
- `Avatar`: Profile picture URL
- `Status`: Real-time user presence status
- `CreatedAt/UpdatedAt`: Audit timestamps

---

#### `/internal/models/room.go` (18 lines)

**Purpose**: Chat room data structure

**Data structure**:

```go
type Room struct {
    ID          primitive.ObjectID    // Unique room identifier
    Name        string                // Room display name
    Description string                // Room description
    CreatorID   primitive.ObjectID    // User who created the room
    Members     []primitive.ObjectID  // List of member user IDs
    IsPrivate   bool                  // Public vs private room flag
    CreatedAt   time.Time             // Room creation timestamp
    UpdatedAt   time.Time             // Last update timestamp
}
```

**Use cases**:
- Public chat rooms (IsPrivate = false)
- Private group chats (IsPrivate = true)
- Direct messages (IsPrivate = true, Members = 2)
- Broadcast channels (IsPrivate = false, limited posting)

**Room features**:
- Member list management
- Access control via IsPrivate flag
- Creator permissions
- Message history per room

---

#### `/internal/models/message.go` (32 lines)

**Purpose**: Message data structures and types

**Data structures**:

```go
type Message struct {
    ID        primitive.ObjectID  // Unique message identifier
    RoomID    primitive.ObjectID  // Room where message was sent
    UserID    primitive.ObjectID  // User who sent the message
    Username  string              // Cached username for display
    Content   string              // Message text content
    Type      MessageType         // Message type (text, join, leave, etc.)
    Timestamp time.Time           // When message was sent
}

type MessageType string

const (
    MessageTypeText   MessageType = "text"    // Regular chat message
    MessageTypeJoin   MessageType = "join"    // User joined notification
    MessageTypeLeave  MessageType = "leave"   // User left notification
    MessageTypeSystem MessageType = "system"  // System notification
)

type ClientMessage struct {
    Type    string  // Message type from client
    RoomID  string  // Target room ID
    Content string  // Message content
}
```

**Message types explained**:

1. **MessageTypeText**: Standard user messages
   - Contains user-generated content
   - Saved to database
   - Broadcast to all room members

2. **MessageTypeJoin**: User join notifications
   - Sent when user enters room
   - Shows "User joined" message
   - Updates room member count

3. **MessageTypeLeave**: User departure notifications
   - Sent when user leaves room
   - Shows "User left" message
   - Updates room member count

4. **MessageTypeSystem**: System messages
   - Server-generated notifications
   - Errors, warnings, announcements
   - Not attributed to specific user

---

#### `/internal/hub/hub.go` (149 lines)

**Purpose**: Central message broker and connection manager

**Core structures**:

```go
type Client struct {
    ID     string              // Unique client connection ID
    UserID string              // Associated user ID
    RoomID string              // Current room ID
    Conn   *websocket.Conn     // WebSocket connection
    Send   chan []byte         // Buffered send channel
}

type Hub struct {
    clients    map[string]*Client                // All clients by ID
    rooms      map[string]map[string]*Client     // Rooms -> Clients
    broadcast  chan *models.Message              // Broadcast channel
    register   chan *Client                      // Registration requests
    unregister chan *Client                      // Unregistration requests
    pubsub     *pubsub.RedisPubSub              // Redis connection
    mu         sync.RWMutex                      // Concurrent access lock
}
```

**Key methods explained**:

1. **New(ps *pubsub.RedisPubSub) *Hub**
   ```go
   // Creates new hub instance
   // Initializes all maps and channels
   // Sets up Redis pub/sub connection
   return &Hub{
       clients:    make(map[string]*Client),
       rooms:      make(map[string]map[string]*Client),
       broadcast:  make(chan *models.Message, 256),  // Buffered channel
       register:   make(chan *Client),
       unregister: make(chan *Client),
       pubsub:     ps,
   }
   ```

2. **Run()**
   ```go
   // Main event loop - runs in goroutine
   // Handles three event types:

   // 1. Client Registration
   case client := <-h.register:
       h.registerClient(client)
       // Adds client to maps
       // Adds to room
       // Logs connection

   // 2. Client Unregistration
   case client := <-h.unregister:
       h.unregisterClient(client)
       // Removes from maps
       // Closes channels
       // Logs disconnection

   // 3. Message Broadcasting
   case message := <-h.broadcast:
       // Marshals message to JSON
       // Publishes to Redis
       // Distributed to all servers
   ```

3. **registerClient(client *Client)**
   ```go
   // Thread-safe client registration
   h.mu.Lock()
   defer h.mu.Unlock()

   // Add to global client map
   h.clients[client.ID] = client

   // Initialize room if needed
   if h.rooms[client.RoomID] == nil {
       h.rooms[client.RoomID] = make(map[string]*Client)
   }

   // Add client to room
   h.rooms[client.RoomID][client.ID] = client
   ```

4. **broadcastToRoom(roomID string, message *models.Message)**
   ```go
   // Sends message to all clients in a room
   h.mu.RLock()
   defer h.mu.RUnlock()

   clients, ok := h.rooms[roomID]
   if !ok {
       return  // Room doesn't exist
   }

   // Marshal message once
   data, _ := json.Marshal(message)

   // Send to each client
   for _, client := range clients {
       select {
       case client.Send <- data:
           // Message queued successfully
       default:
           // Client buffer full - clean up
           close(client.Send)
           delete(h.clients, client.ID)
           delete(h.rooms[roomID], client.ID)
       }
   }
   ```

**Concurrency patterns**:
- RWMutex for read-heavy operations (broadcasting)
- Buffered channels to prevent blocking
- Select with default for non-blocking sends
- Separate goroutine for each client connection

---

#### `/internal/handlers/websocket.go` (149 lines)

**Purpose**: WebSocket connection upgrade and message handling

**Core components**:

```go
var upgrader = websocket.Upgrader{
    ReadBufferSize:  1024,      // 1KB read buffer
    WriteBufferSize: 1024,      // 1KB write buffer
    CheckOrigin: func(r *http.Request) bool {
        return true  // Configure for production!
    },
}

type WebSocketHandler struct {
    hub     *hub.Hub                      // Reference to hub
    msgRepo repository.MessageRepository  // Database access
}
```

**Key methods**:

1. **ServeWS(w http.ResponseWriter, r *http.Request)**
   ```go
   // HTTP to WebSocket upgrade process

   // Step 1: Upgrade connection
   conn, err := upgrader.Upgrade(w, r, nil)
   // Negotiates WebSocket protocol
   // Returns WebSocket connection

   // Step 2: Extract parameters
   userID := r.URL.Query().Get("user_id")
   roomID := r.URL.Query().Get("room_id")
   // Get from query params or JWT claims

   // Step 3: Create client
   client := &hub.Client{
       ID:     uuid.New().String(),  // Unique connection ID
       UserID: userID,
       RoomID: roomID,
       Conn:   conn,
       Send:   make(chan []byte, 256),  // Buffered send channel
   }

   // Step 4: Register with hub
   h.hub.Register(client)

   // Step 5: Start pumps
   go h.writePump(client)  // Server → Client
   go h.readPump(client)   // Client → Server
   ```

2. **readPump(client *hub.Client)**
   ```go
   // Reads messages from client WebSocket

   // Setup
   defer func() {
       h.hub.Unregister(client)  // Cleanup on exit
       client.Conn.Close()
   }()

   // Configure deadlines
   client.Conn.SetReadDeadline(time.Now().Add(60 * time.Second))

   // Pong handler - resets read deadline
   client.Conn.SetPongHandler(func(string) error {
       client.Conn.SetReadDeadline(time.Now().Add(60 * time.Second))
       return nil
   })

   // Message loop
   for {
       // Read message
       _, msgData, err := client.Conn.ReadMessage()
       if err != nil {
           break  // Connection closed
       }

       // Parse message
       var clientMsg models.ClientMessage
       json.Unmarshal(msgData, &clientMsg)

       // Create message model
       message := &models.Message{
           RoomID:    roomID,
           UserID:    userID,
           Content:   clientMsg.Content,
           Type:      models.MessageTypeText,
           Timestamp: time.Now(),
       }

       // Save to database
       h.msgRepo.Create(message)

       // Broadcast to hub
       h.hub.Broadcast(message)
   }
   ```

3. **writePump(client *hub.Client)**
   ```go
   // Sends messages to client WebSocket

   // Setup heartbeat ticker
   ticker := time.NewTicker(54 * time.Second)
   defer ticker.Stop()

   for {
       select {
       // Message from hub
       case message, ok := <-client.Send:
           client.Conn.SetWriteDeadline(time.Now().Add(10 * time.Second))

           if !ok {
               // Channel closed
               client.Conn.WriteMessage(websocket.CloseMessage, []byte{})
               return
           }

           // Write message
           w, _ := client.Conn.NextWriter(websocket.TextMessage)
           w.Write(message)
           w.Close()

       // Heartbeat tick
       case <-ticker.C:
           client.Conn.SetWriteDeadline(time.Now().Add(10 * time.Second))
           // Send ping to client
           if err := client.Conn.WriteMessage(websocket.PingMessage, nil); err != nil {
               return  // Connection dead
           }
       }
   }
   ```

**WebSocket message flow**:
```
Client sends → readPump receives → Parse JSON → Save to MongoDB
    → Send to Hub → Hub publishes to Redis → All servers receive
    → broadcastToRoom → writePump sends → Client receives
```

**Connection health monitoring**:
- Ping sent every 54 seconds
- Pong expected within 60 seconds
- Read deadline reset on each pong
- Connection closed if deadline exceeded

---

#### `/internal/pubsub/redis.go` (45 lines)

**Purpose**: Redis pub/sub wrapper for cross-server messaging

**Implementation**:

```go
type RedisPubSub struct {
    client *redis.Client      // Redis client connection
    ctx    context.Context    // Context for operations
}

func New(addr string) *RedisPubSub {
    // Create Redis client
    client := redis.NewClient(&redis.Options{
        Addr: addr,  // e.g., "localhost:6379"
    })

    return &RedisPubSub{
        client: client,
        ctx:    context.Background(),
    }
}

func (r *RedisPubSub) Publish(channel, message string) error {
    // Publish message to Redis channel
    // All subscribed servers receive the message
    return r.client.Publish(r.ctx, channel, message).Err()
}

func (r *RedisPubSub) Subscribe(channel string, handler func(string)) {
    // Subscribe to Redis channel
    pubsub := r.client.Subscribe(r.ctx, channel)
    defer pubsub.Close()

    // Get channel for messages
    ch := pubsub.Channel()

    // Process messages in goroutine
    go func() {
        for msg := range ch {
            handler(msg.Payload)  // Call handler for each message
        }
    }()
}
```

**Multi-server scenario**:

```
Server 1: Client A sends message
    ↓
Server 1: Publishes to Redis channel "chat"
    ↓
Redis: Broadcasts to all subscribers
    ↓
Server 1: Receives message → Broadcasts to local clients
Server 2: Receives message → Broadcasts to local clients
Server 3: Receives message → Broadcasts to local clients
```

**Benefits**:
- Stateless servers
- No server affinity needed
- Easy horizontal scaling
- Automatic failover

---

#### `/internal/repository/message_repository.go` (70 lines)

**Purpose**: MongoDB message persistence layer

**Interface definition**:

```go
type MessageRepository interface {
    Create(message *models.Message) error
    GetByRoom(roomID primitive.ObjectID, limit int) ([]*models.Message, error)
    Delete(id primitive.ObjectID) error
}
```

**Implementation**:

```go
type messageRepository struct {
    collection *mongo.Collection  // MongoDB collection
}

func NewMessageRepository(db *mongo.Database) MessageRepository {
    return &messageRepository{
        collection: db.Collection("messages"),
    }
}
```

**Methods explained**:

1. **Create(message *models.Message)**
   ```go
   // Saves message to MongoDB

   // Create context with timeout
   ctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
   defer cancel()

   // Generate new ObjectID
   message.ID = primitive.NewObjectID()

   // Insert into collection
   _, err := r.collection.InsertOne(ctx, message)
   return err
   ```

2. **GetByRoom(roomID primitive.ObjectID, limit int)**
   ```go
   // Retrieves recent messages for a room

   ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
   defer cancel()

   // Configure query options
   opts := options.Find().
       SetSort(bson.D{{Key: "timestamp", Value: -1}}).  // Newest first
       SetLimit(int64(limit))                            // Limit results

   // Find messages for room
   filter := bson.M{"room_id": roomID}
   cursor, err := r.collection.Find(ctx, filter, opts)

   // Decode results
   var messages []*models.Message
   cursor.All(ctx, &messages)

   return messages, nil
   ```

3. **Delete(id primitive.ObjectID)**
   ```go
   // Deletes a message (admin/moderation)

   ctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
   defer cancel()

   filter := bson.M{"_id": id}
   _, err := r.collection.DeleteOne(ctx, filter)
   return err
   ```

**Database indexes** (should be created):

```javascript
// MongoDB shell commands
db.messages.createIndex({ "room_id": 1, "timestamp": -1 })
db.messages.createIndex({ "user_id": 1, "timestamp": -1 })
db.messages.createIndex({ "timestamp": 1 })
```

---

#### `/internal/auth/jwt.go` (50 lines)

**Purpose**: JWT token generation and validation

**Implementation**:

```go
var secretKey = []byte("your-secret-key")  // Use env var in production

type Claims struct {
    UserID   string  // User identifier
    Username string  // User display name
    jwt.RegisteredClaims
}

func GenerateToken(userID, username string) (string, error) {
    // Create claims
    claims := &Claims{
        UserID:   userID,
        Username: username,
        RegisteredClaims: jwt.RegisteredClaims{
            ExpiresAt: jwt.NewNumericDate(time.Now().Add(24 * time.Hour)),
            IssuedAt:  jwt.NewNumericDate(time.Now()),
        },
    }

    // Create token
    token := jwt.NewWithClaims(jwt.SigningMethodHS256, claims)

    // Sign and return
    return token.SignedString(secretKey)
}

func ValidateToken(tokenString string) (*Claims, error) {
    // Parse token
    token, err := jwt.ParseWithClaims(
        tokenString,
        &Claims{},
        func(token *jwt.Token) (interface{}, error) {
            return secretKey, nil
        },
    )

    if err != nil {
        return nil, err
    }

    // Extract and validate claims
    if claims, ok := token.Claims.(*Claims); ok && token.Valid {
        return claims, nil
    }

    return nil, ErrInvalidToken
}
```

**Usage in middleware**:

```go
func AuthMiddleware(next http.Handler) http.Handler {
    return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
        // Get token from header
        authHeader := r.Header.Get("Authorization")
        tokenString := strings.TrimPrefix(authHeader, "Bearer ")

        // Validate token
        claims, err := auth.ValidateToken(tokenString)
        if err != nil {
            http.Error(w, "Unauthorized", http.StatusUnauthorized)
            return
        }

        // Add claims to context
        ctx := context.WithValue(r.Context(), "user_id", claims.UserID)
        ctx = context.WithValue(ctx, "username", claims.Username)

        // Call next handler
        next.ServeHTTP(w, r.WithContext(ctx))
    })
}
```

---

## Quick Start

### Prerequisites

- **Go 1.24 or higher**
  ```bash
  go version
  ```

- **Docker and Docker Compose**
  ```bash
  docker --version
  docker-compose --version
  ```

### Option 1: Docker Compose (Recommended)

Start all services with a single command:

```bash
# Start all services (app, MongoDB, Redis)
docker-compose up -d

# View logs
docker-compose logs -f app

# View all services
docker-compose ps

# Stop all services
docker-compose down

# Stop and remove volumes (clean slate)
docker-compose down -v
```

The server will be available at `ws://localhost:8080/ws`

### Option 2: Local Development

Requires MongoDB and Redis running locally.

```bash
# Start MongoDB (separate terminal)
docker run -d -p 27017:27017 mongo:7

# Start Redis (separate terminal)
docker run -d -p 6379:6379 redis:7-alpine

# Install dependencies
go mod download

# Run the server
go run cmd/server/main.go

# Or build and run
go build -o bin/server cmd/server/main.go
./bin/server
```

### Option 3: Production Build

Build optimized binary:

```bash
# Build with optimizations
CGO_ENABLED=0 GOOS=linux GOARCH=amd64 go build \
    -ldflags="-w -s" \
    -o bin/server \
    cmd/server/main.go

# Run
./bin/server
```

Build flags explained:
- `CGO_ENABLED=0`: Static binary (no C dependencies)
- `-ldflags="-w -s"`: Strip debug info (-w) and symbol table (-s)
- Reduces binary size by ~30%

---

## Implementation Guide

### Step 1: Project Setup

1. **Initialize Go module**
   ```bash
   mkdir chatserver
   cd chatserver
   go mod init github.com/yourusername/chatserver
   ```

2. **Create directory structure**
   ```bash
   mkdir -p cmd/server
   mkdir -p internal/{auth,handlers,hub,models,pubsub,repository}
   ```

3. **Install dependencies**
   ```bash
   go get github.com/go-chi/chi/v5
   go get github.com/go-chi/cors
   go get github.com/golang-jwt/jwt/v5
   go get github.com/google/uuid
   go get github.com/gorilla/websocket
   go get github.com/redis/go-redis/v9
   go get go.mongodb.org/mongo-driver/mongo
   ```

### Step 2: Implement Core Models

Start with data structures in `internal/models/`:

1. **Create user.go**
   ```go
   package models

   import (
       "time"
       "go.mongodb.org/mongo-driver/bson/primitive"
   )

   type User struct {
       ID        primitive.ObjectID `bson:"_id,omitempty" json:"id"`
       Username  string             `bson:"username" json:"username"`
       Email     string             `bson:"email" json:"email"`
       Password  string             `bson:"password" json:"-"`
       Avatar    string             `bson:"avatar" json:"avatar"`
       Status    UserStatus         `bson:"status" json:"status"`
       CreatedAt time.Time          `bson:"created_at" json:"created_at"`
       UpdatedAt time.Time          `bson:"updated_at" json:"updated_at"`
   }

   type UserStatus string

   const (
       StatusOnline  UserStatus = "online"
       StatusOffline UserStatus = "offline"
       StatusAway    UserStatus = "away"
   )
   ```

2. **Create message.go and room.go** (see file structure above)

### Step 3: Implement Redis Pub/Sub

Create `internal/pubsub/redis.go`:

```go
package pubsub

import (
    "context"
    "github.com/redis/go-redis/v9"
)

type RedisPubSub struct {
    client *redis.Client
    ctx    context.Context
}

func New(addr string) *RedisPubSub {
    client := redis.NewClient(&redis.Options{
        Addr: addr,
        // Add password, DB, etc. for production
    })

    return &RedisPubSub{
        client: client,
        ctx:    context.Background(),
    }
}

func (r *RedisPubSub) Publish(channel, message string) error {
    return r.client.Publish(r.ctx, channel, message).Err()
}

func (r *RedisPubSub) Subscribe(channel string, handler func(string)) {
    pubsub := r.client.Subscribe(r.ctx, channel)
    defer pubsub.Close()

    ch := pubsub.Channel()

    go func() {
        for msg := range ch {
            handler(msg.Payload)
        }
    }()
}

func (r *RedisPubSub) Close() error {
    return r.client.Close()
}
```

### Step 4: Implement Hub

Create `internal/hub/hub.go` - the core of the messaging system:

```go
package hub

import (
    "encoding/json"
    "log"
    "sync"

    "github.com/gorilla/websocket"
    "github.com/yourusername/chatserver/internal/models"
    "github.com/yourusername/chatserver/internal/pubsub"
)

type Client struct {
    ID     string
    UserID string
    RoomID string
    Conn   *websocket.Conn
    Send   chan []byte
}

type Hub struct {
    clients    map[string]*Client
    rooms      map[string]map[string]*Client
    broadcast  chan *models.Message
    register   chan *Client
    unregister chan *Client
    pubsub     *pubsub.RedisPubSub
    mu         sync.RWMutex
}

func New(ps *pubsub.RedisPubSub) *Hub {
    return &Hub{
        clients:    make(map[string]*Client),
        rooms:      make(map[string]map[string]*Client),
        broadcast:  make(chan *models.Message, 256),
        register:   make(chan *Client),
        unregister: make(chan *Client),
        pubsub:     ps,
    }
}

func (h *Hub) Run() {
    // Subscribe to Redis messages
    h.pubsub.Subscribe("chat", func(message string) {
        var msg models.Message
        if err := json.Unmarshal([]byte(message), &msg); err != nil {
            log.Printf("Error unmarshaling: %v", err)
            return
        }
        h.broadcastToRoom(msg.RoomID.Hex(), &msg)
    })

    // Main event loop
    for {
        select {
        case client := <-h.register:
            h.registerClient(client)
        case client := <-h.unregister:
            h.unregisterClient(client)
        case message := <-h.broadcast:
            data, _ := json.Marshal(message)
            h.pubsub.Publish("chat", string(data))
        }
    }
}

// Additional methods: registerClient, unregisterClient, broadcastToRoom, etc.
```

### Step 5: Implement WebSocket Handler

Create `internal/handlers/websocket.go`:

```go
package handlers

import (
    "encoding/json"
    "log"
    "net/http"
    "time"

    "github.com/google/uuid"
    "github.com/gorilla/websocket"
    "github.com/yourusername/chatserver/internal/hub"
    "github.com/yourusername/chatserver/internal/models"
    "github.com/yourusername/chatserver/internal/repository"
)

var upgrader = websocket.Upgrader{
    ReadBufferSize:  1024,
    WriteBufferSize: 1024,
    CheckOrigin: func(r *http.Request) bool {
        // Configure properly for production
        return true
    },
}

type WebSocketHandler struct {
    hub     *hub.Hub
    msgRepo repository.MessageRepository
}

func NewWebSocketHandler(h *hub.Hub, msgRepo repository.MessageRepository) *WebSocketHandler {
    return &WebSocketHandler{hub: h, msgRepo: msgRepo}
}

func (h *WebSocketHandler) ServeWS(w http.ResponseWriter, r *http.Request) {
    // Upgrade connection
    conn, err := upgrader.Upgrade(w, r, nil)
    if err != nil {
        log.Printf("Upgrade error: %v", err)
        return
    }

    // Get user/room from query or JWT
    userID := r.URL.Query().Get("user_id")
    roomID := r.URL.Query().Get("room_id")

    if userID == "" || roomID == "" {
        conn.Close()
        return
    }

    // Create client
    client := &hub.Client{
        ID:     uuid.New().String(),
        UserID: userID,
        RoomID: roomID,
        Conn:   conn,
        Send:   make(chan []byte, 256),
    }

    h.hub.Register(client)

    // Start read/write pumps
    go h.writePump(client)
    go h.readPump(client)
}

// Implement readPump and writePump
```

### Step 6: Implement Repository

Create `internal/repository/message_repository.go`:

```go
package repository

import (
    "context"
    "time"

    "github.com/yourusername/chatserver/internal/models"
    "go.mongodb.org/mongo-driver/bson"
    "go.mongodb.org/mongo-driver/bson/primitive"
    "go.mongodb.org/mongo-driver/mongo"
    "go.mongodb.org/mongo-driver/mongo/options"
)

type MessageRepository interface {
    Create(message *models.Message) error
    GetByRoom(roomID primitive.ObjectID, limit int) ([]*models.Message, error)
    Delete(id primitive.ObjectID) error
}

type messageRepository struct {
    collection *mongo.Collection
}

func NewMessageRepository(db *mongo.Database) MessageRepository {
    return &messageRepository{
        collection: db.Collection("messages"),
    }
}

// Implement Create, GetByRoom, Delete methods
```

### Step 7: Create Main Application

Create `cmd/server/main.go`:

```go
package main

import (
    "context"
    "log"
    "net/http"
    "os"
    "os/signal"
    "syscall"
    "time"

    "github.com/go-chi/chi/v5"
    "github.com/go-chi/chi/v5/middleware"
    "github.com/go-chi/cors"
    "github.com/yourusername/chatserver/internal/handlers"
    "github.com/yourusername/chatserver/internal/hub"
    "github.com/yourusername/chatserver/internal/pubsub"
    "github.com/yourusername/chatserver/internal/repository"
    "go.mongodb.org/mongo-driver/mongo"
    "go.mongodb.org/mongo-driver/mongo/options"
)

func main() {
    // Initialize MongoDB
    ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
    defer cancel()

    mongoURI := getEnv("MONGO_URI", "mongodb://localhost:27017")
    client, err := mongo.Connect(ctx, options.Client().ApplyURI(mongoURI))
    if err != nil {
        log.Fatal(err)
    }
    defer client.Disconnect(ctx)

    db := client.Database("chatserver")

    // Initialize Redis
    redisAddr := getEnv("REDIS_ADDR", "localhost:6379")
    ps := pubsub.New(redisAddr)

    // Initialize Hub
    h := hub.New(ps)
    go h.Run()

    // Initialize repositories and handlers
    msgRepo := repository.NewMessageRepository(db)
    wsHandler := handlers.NewWebSocketHandler(h, msgRepo)

    // Setup router
    r := chi.NewRouter()
    r.Use(middleware.Logger)
    r.Use(middleware.Recoverer)
    r.Use(cors.Handler(cors.Options{
        AllowedOrigins: []string{"http://localhost:3000"},
        AllowedMethods: []string{"GET", "POST", "PUT", "DELETE", "OPTIONS"},
        AllowedHeaders: []string{"Accept", "Authorization", "Content-Type"},
    }))

    r.Get("/ws", wsHandler.ServeWS)
    r.Get("/health", func(w http.ResponseWriter, r *http.Request) {
        w.Write([]byte("OK"))
    })

    // Start server with graceful shutdown
    port := getEnv("PORT", "8080")
    srv := &http.Server{Addr: ":" + port, Handler: r}

    go func() {
        log.Printf("Server starting on port %s", port)
        if err := srv.ListenAndServe(); err != http.ErrServerClosed {
            log.Fatal(err)
        }
    }()

    // Wait for interrupt
    quit := make(chan os.Signal, 1)
    signal.Notify(quit, syscall.SIGINT, syscall.SIGTERM)
    <-quit

    log.Println("Shutting down...")
    ctx, cancel = context.WithTimeout(context.Background(), 5*time.Second)
    defer cancel()
    srv.Shutdown(ctx)
}

func getEnv(key, fallback string) string {
    if value := os.Getenv(key); value != "" {
        return value
    }
    return fallback
}
```

### Step 8: Create Dockerfile

```dockerfile
FROM golang:1.24-alpine AS builder

WORKDIR /app

COPY go.mod go.sum* ./
RUN go mod download && go mod verify

COPY . .

RUN CGO_ENABLED=0 GOOS=linux GOARCH=amd64 go build \
    -ldflags="-w -s" \
    -o /app/bin/server \
    ./cmd/server/main.go

FROM alpine:latest

RUN apk --no-cache add ca-certificates

WORKDIR /app

RUN adduser -D -u 1000 appuser
USER appuser

COPY --from=builder /app/bin/server /app/server

EXPOSE 8080

CMD ["/app/server"]
```

### Step 9: Create Docker Compose

```yaml
version: '3.8'

services:
  app:
    build: .
    ports:
      - "8080:8080"
    environment:
      - MONGO_URI=mongodb://mongo:27017
      - REDIS_ADDR=redis:6379
      - PORT=8080
    depends_on:
      - mongo
      - redis
    networks:
      - chatserver-network

  mongo:
    image: mongo:7
    ports:
      - "27017:27017"
    volumes:
      - mongo-data:/data/db
    networks:
      - chatserver-network

  redis:
    image: redis:7-alpine
    ports:
      - "6379:6379"
    networks:
      - chatserver-network

networks:
  chatserver-network:

volumes:
  mongo-data:
```

### Step 10: Test the Implementation

1. **Start services**
   ```bash
   docker-compose up -d
   ```

2. **Test WebSocket connection** (using browser console or tool like Postman):
   ```javascript
   const ws = new WebSocket('ws://localhost:8080/ws?user_id=user123&room_id=room456');

   ws.onopen = () => {
       console.log('Connected');
       ws.send(JSON.stringify({
           type: 'message',
           room_id: 'room456',
           content: 'Hello, World!'
       }));
   };

   ws.onmessage = (event) => {
       console.log('Received:', JSON.parse(event.data));
   };
   ```

3. **Check MongoDB**
   ```bash
   docker exec -it chatserver-mongo-1 mongosh
   use chatserver
   db.messages.find().pretty()
   ```

4. **Check Redis**
   ```bash
   docker exec -it chatserver-redis-1 redis-cli
   MONITOR
   ```

---

## Testing Strategy

### Unit Tests

#### Testing the Hub

Create `internal/hub/hub_test.go`:

```go
package hub

import (
    "testing"
    "time"

    "github.com/gorilla/websocket"
    "github.com/stretchr/testify/assert"
    "github.com/yourusername/chatserver/internal/pubsub"
)

func TestHubRegisterClient(t *testing.T) {
    ps := pubsub.New("localhost:6379")
    h := New(ps)

    client := &Client{
        ID:     "test-client-1",
        UserID: "user-1",
        RoomID: "room-1",
        Send:   make(chan []byte, 256),
    }

    h.Register(client)

    // Give time for registration
    time.Sleep(100 * time.Millisecond)

    assert.Equal(t, 1, h.GetRoomClients("room-1"))
}

func TestHubUnregisterClient(t *testing.T) {
    ps := pubsub.New("localhost:6379")
    h := New(ps)

    client := &Client{
        ID:     "test-client-1",
        UserID: "user-1",
        RoomID: "room-1",
        Send:   make(chan []byte, 256),
    }

    h.Register(client)
    time.Sleep(100 * time.Millisecond)

    h.Unregister(client)
    time.Sleep(100 * time.Millisecond)

    assert.Equal(t, 0, h.GetRoomClients("room-1"))
}
```

#### Testing Message Repository

Create `internal/repository/message_repository_test.go`:

```go
package repository

import (
    "context"
    "testing"
    "time"

    "github.com/stretchr/testify/assert"
    "github.com/yourusername/chatserver/internal/models"
    "go.mongodb.org/mongo-driver/bson/primitive"
    "go.mongodb.org/mongo-driver/mongo"
    "go.mongodb.org/mongo-driver/mongo/options"
)

func setupTestDB(t *testing.T) *mongo.Database {
    ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
    defer cancel()

    client, err := mongo.Connect(ctx, options.Client().ApplyURI("mongodb://localhost:27017"))
    if err != nil {
        t.Fatal(err)
    }

    return client.Database("chatserver_test")
}

func TestCreateMessage(t *testing.T) {
    db := setupTestDB(t)
    repo := NewMessageRepository(db)

    message := &models.Message{
        RoomID:    primitive.NewObjectID(),
        UserID:    primitive.NewObjectID(),
        Content:   "Test message",
        Type:      models.MessageTypeText,
        Timestamp: time.Now(),
    }

    err := repo.Create(message)
    assert.NoError(t, err)
    assert.NotEmpty(t, message.ID)
}

func TestGetByRoom(t *testing.T) {
    db := setupTestDB(t)
    repo := NewMessageRepository(db)

    roomID := primitive.NewObjectID()

    // Create test messages
    for i := 0; i < 5; i++ {
        message := &models.Message{
            RoomID:    roomID,
            UserID:    primitive.NewObjectID(),
            Content:   "Test message",
            Type:      models.MessageTypeText,
            Timestamp: time.Now(),
        }
        repo.Create(message)
    }

    // Retrieve messages
    messages, err := repo.GetByRoom(roomID, 10)
    assert.NoError(t, err)
    assert.Len(t, messages, 5)
}
```

### Integration Tests

Create `test/integration_test.go`:

```go
package test

import (
    "encoding/json"
    "net/http"
    "net/http/httptest"
    "strings"
    "testing"
    "time"

    "github.com/gorilla/websocket"
    "github.com/stretchr/testify/assert"
)

func TestWebSocketConnection(t *testing.T) {
    // Setup test server
    server := setupTestServer()
    defer server.Close()

    // Connect to WebSocket
    wsURL := "ws" + strings.TrimPrefix(server.URL, "http") + "/ws?user_id=test&room_id=test"
    ws, _, err := websocket.DefaultDialer.Dial(wsURL, nil)
    assert.NoError(t, err)
    defer ws.Close()

    // Send message
    msg := map[string]string{
        "type":    "message",
        "room_id": "test",
        "content": "Hello",
    }
    data, _ := json.Marshal(msg)
    err = ws.WriteMessage(websocket.TextMessage, data)
    assert.NoError(t, err)

    // Receive message
    ws.SetReadDeadline(time.Now().Add(5 * time.Second))
    _, response, err := ws.ReadMessage()
    assert.NoError(t, err)

    var received map[string]interface{}
    json.Unmarshal(response, &received)
    assert.Equal(t, "Hello", received["content"])
}
```

### Load Testing

Use a tool like `k6` for load testing:

Create `test/load.js`:

```javascript
import ws from 'k6/ws';
import { check } from 'k6';

export let options = {
    stages: [
        { duration: '30s', target: 100 },  // Ramp up to 100 users
        { duration: '1m', target: 100 },   // Stay at 100 users
        { duration: '30s', target: 0 },    // Ramp down to 0
    ],
};

export default function () {
    const url = 'ws://localhost:8080/ws?user_id=user&room_id=room';

    const res = ws.connect(url, function (socket) {
        socket.on('open', function () {
            socket.send(JSON.stringify({
                type: 'message',
                room_id: 'room',
                content: 'Test message'
            }));
        });

        socket.on('message', function (msg) {
            check(msg, { 'message received': (m) => m.length > 0 });
        });

        socket.setTimeout(function () {
            socket.close();
        }, 60000);
    });

    check(res, { 'status is 101': (r) => r && r.status === 101 });
}
```

Run load test:
```bash
k6 run test/load.js
```

### Coverage Goals

Aim for these coverage targets:

- **Overall coverage**: 70%+
- **Core packages**: 80%+
  - `internal/hub`: 85%+
  - `internal/handlers`: 80%+
  - `internal/repository`: 90%+
- **Utility packages**: 60%+

Generate coverage report:

```bash
# Run tests with coverage
go test -coverprofile=coverage.out ./...

# Generate HTML report
go tool cover -html=coverage.out -o coverage.html

# View coverage by package
go tool cover -func=coverage.out
```

---

## Deployment Guide

### Development Deployment

Use Docker Compose for local development:

```bash
# Start services
docker-compose up -d

# View logs
docker-compose logs -f

# Restart a service
docker-compose restart app

# Stop services
docker-compose down
```

### Production Deployment

#### Option 1: Docker Swarm

1. **Initialize Swarm**
   ```bash
   docker swarm init
   ```

2. **Create Stack File** (`docker-stack.yml`):
   ```yaml
   version: '3.8'

   services:
     app:
       image: yourregistry/chatserver:latest
       deploy:
         replicas: 3
         update_config:
           parallelism: 1
           delay: 10s
         restart_policy:
           condition: on-failure
       ports:
         - "8080:8080"
       environment:
         - MONGO_URI=mongodb://mongo:27017
         - REDIS_ADDR=redis:6379
       networks:
         - chatserver-network

     mongo:
       image: mongo:7
       deploy:
         replicas: 1
         placement:
           constraints:
             - node.role == manager
       volumes:
         - mongo-data:/data/db
       networks:
         - chatserver-network

     redis:
       image: redis:7-alpine
       deploy:
         replicas: 1
       networks:
         - chatserver-network

   networks:
     chatserver-network:
       driver: overlay

   volumes:
     mongo-data:
   ```

3. **Deploy Stack**
   ```bash
   docker stack deploy -c docker-stack.yml chatserver

   # View services
   docker service ls

   # View logs
   docker service logs chatserver_app

   # Scale service
   docker service scale chatserver_app=5
   ```

#### Option 2: Kubernetes

1. **Create Deployment** (`k8s/deployment.yaml`):
   ```yaml
   apiVersion: apps/v1
   kind: Deployment
   metadata:
     name: chatserver
   spec:
     replicas: 3
     selector:
       matchLabels:
         app: chatserver
     template:
       metadata:
         labels:
           app: chatserver
       spec:
         containers:
         - name: chatserver
           image: yourregistry/chatserver:latest
           ports:
           - containerPort: 8080
           env:
           - name: MONGO_URI
             value: mongodb://mongo:27017
           - name: REDIS_ADDR
             value: redis:6379
           resources:
             requests:
               memory: "128Mi"
               cpu: "250m"
             limits:
               memory: "512Mi"
               cpu: "500m"
   ```

2. **Create Service** (`k8s/service.yaml`):
   ```yaml
   apiVersion: v1
   kind: Service
   metadata:
     name: chatserver
   spec:
     type: LoadBalancer
     ports:
     - port: 80
       targetPort: 8080
       protocol: TCP
     selector:
       app: chatserver
   ```

3. **Deploy to Kubernetes**
   ```bash
   kubectl apply -f k8s/deployment.yaml
   kubectl apply -f k8s/service.yaml

   # View pods
   kubectl get pods

   # View logs
   kubectl logs -f deployment/chatserver

   # Scale deployment
   kubectl scale deployment chatserver --replicas=5
   ```

#### Option 3: AWS ECS

1. **Create Task Definition** (`ecs-task.json`):
   ```json
   {
     "family": "chatserver",
     "containerDefinitions": [
       {
         "name": "chatserver",
         "image": "yourregistry/chatserver:latest",
         "memory": 512,
         "cpu": 256,
         "essential": true,
         "portMappings": [
           {
             "containerPort": 8080,
             "protocol": "tcp"
           }
         ],
         "environment": [
           {
             "name": "MONGO_URI",
             "value": "mongodb://your-mongo-host:27017"
           },
           {
             "name": "REDIS_ADDR",
             "value": "your-redis-host:6379"
           }
         ]
       }
     ]
   }
   ```

2. **Deploy to ECS**
   ```bash
   # Register task definition
   aws ecs register-task-definition --cli-input-json file://ecs-task.json

   # Create service
   aws ecs create-service \
     --cluster your-cluster \
     --service-name chatserver \
     --task-definition chatserver \
     --desired-count 3 \
     --launch-type FARGATE
   ```

### Environment Variables

Set these for production:

```bash
# Server
PORT=8080

# MongoDB
MONGO_URI=mongodb://user:pass@host:27017/chatserver?authSource=admin

# Redis
REDIS_ADDR=redis-host:6379
REDIS_PASSWORD=yourpassword
REDIS_DB=0

# JWT
JWT_SECRET=your-super-secret-key-change-this

# Logging
LOG_LEVEL=info
LOG_FORMAT=json
```

### SSL/TLS Configuration

Use a reverse proxy like nginx for SSL termination:

```nginx
upstream chatserver {
    server app1:8080;
    server app2:8080;
    server app3:8080;
}

server {
    listen 443 ssl http2;
    server_name chat.example.com;

    ssl_certificate /etc/nginx/ssl/cert.pem;
    ssl_certificate_key /etc/nginx/ssl/key.pem;

    location /ws {
        proxy_pass http://chatserver;
        proxy_http_version 1.1;
        proxy_set_header Upgrade $http_upgrade;
        proxy_set_header Connection "upgrade";
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto $scheme;

        # WebSocket timeouts
        proxy_read_timeout 3600s;
        proxy_send_timeout 3600s;
    }

    location /health {
        proxy_pass http://chatserver;
    }
}
```

### Monitoring and Observability

1. **Prometheus Metrics**

Add metrics endpoint:

```go
import "github.com/prometheus/client_golang/prometheus/promhttp"

r.Handle("/metrics", promhttp.Handler())
```

2. **Health Checks**

Implement comprehensive health check:

```go
func healthCheck(w http.ResponseWriter, r *http.Request) {
    // Check MongoDB
    ctx, cancel := context.WithTimeout(context.Background(), 2*time.Second)
    defer cancel()

    if err := mongoClient.Ping(ctx, nil); err != nil {
        http.Error(w, "MongoDB unhealthy", http.StatusServiceUnavailable)
        return
    }

    // Check Redis
    if err := redisClient.Ping(ctx).Err(); err != nil {
        http.Error(w, "Redis unhealthy", http.StatusServiceUnavailable)
        return
    }

    w.WriteHeader(http.StatusOK)
    json.NewEncoder(w).Encode(map[string]string{
        "status": "healthy",
        "timestamp": time.Now().Format(time.RFC3339),
    })
}
```

3. **Structured Logging**

Use structured logging for better observability:

```go
import "go.uber.org/zap"

logger, _ := zap.NewProduction()
defer logger.Sync()

logger.Info("Client connected",
    zap.String("client_id", client.ID),
    zap.String("user_id", client.UserID),
    zap.String("room_id", client.RoomID),
)
```

---

## Performance Optimization

### Connection Pooling

Configure MongoDB connection pool:

```go
clientOptions := options.Client().
    ApplyURI(mongoURI).
    SetMaxPoolSize(100).
    SetMinPoolSize(10).
    SetMaxConnIdleTime(30 * time.Second)
```

Configure Redis connection pool:

```go
redisClient := redis.NewClient(&redis.Options{
    Addr:         redisAddr,
    PoolSize:     100,
    MinIdleConns: 10,
    MaxRetries:   3,
})
```

### Message Batching

Batch messages for improved throughput:

```go
type MessageBatcher struct {
    messages chan *models.Message
    batch    []*models.Message
    ticker   *time.Ticker
}

func (b *MessageBatcher) Start() {
    b.ticker = time.NewTicker(100 * time.Millisecond)

    for {
        select {
        case msg := <-b.messages:
            b.batch = append(b.batch, msg)
            if len(b.batch) >= 100 {
                b.flush()
            }
        case <-b.ticker.C:
            if len(b.batch) > 0 {
                b.flush()
            }
        }
    }
}

func (b *MessageBatcher) flush() {
    // Batch insert to MongoDB
    repo.CreateBatch(b.batch)
    b.batch = b.batch[:0]
}
```

### Compression

Enable WebSocket compression:

```go
var upgrader = websocket.Upgrader{
    ReadBufferSize:  1024,
    WriteBufferSize: 1024,
    EnableCompression: true,  // Enable compression
}
```

### Caching

Cache frequently accessed data:

```go
import "github.com/patrickmn/go-cache"

// Create cache
c := cache.New(5*time.Minute, 10*time.Minute)

// Get room info with caching
func getRoomInfo(roomID string) (*models.Room, error) {
    // Check cache
    if cached, found := c.Get(roomID); found {
        return cached.(*models.Room), nil
    }

    // Fetch from database
    room, err := repo.GetRoom(roomID)
    if err != nil {
        return nil, err
    }

    // Store in cache
    c.Set(roomID, room, cache.DefaultExpiration)
    return room, nil
}
```

### Profiling

Profile CPU and memory usage:

```go
import (
    "net/http"
    _ "net/http/pprof"
)

// Enable pprof endpoint
go func() {
    log.Println(http.ListenAndServe("localhost:6060", nil))
}()
```

Access profiling data:
```bash
# CPU profile
go tool pprof http://localhost:6060/debug/pprof/profile

# Memory profile
go tool pprof http://localhost:6060/debug/pprof/heap

# Goroutine profile
go tool pprof http://localhost:6060/debug/pprof/goroutine
```

---

## Security Considerations

### Origin Validation

Properly configure CheckOrigin:

```go
var upgrader = websocket.Upgrader{
    CheckOrigin: func(r *http.Request) bool {
        origin := r.Header.Get("Origin")
        allowedOrigins := []string{
            "https://yourdomain.com",
            "https://app.yourdomain.com",
        }
        for _, allowed := range allowedOrigins {
            if origin == allowed {
                return true
            }
        }
        return false
    },
}
```

### Input Validation

Validate all client inputs:

```go
func validateMessage(msg *models.ClientMessage) error {
    if len(msg.Content) == 0 {
        return errors.New("content cannot be empty")
    }
    if len(msg.Content) > 5000 {
        return errors.New("content too long")
    }
    if msg.RoomID == "" {
        return errors.New("room_id required")
    }
    return nil
}
```

### Rate Limiting

Implement rate limiting per connection:

```go
import "golang.org/x/time/rate"

type Client struct {
    // ... existing fields
    limiter *rate.Limiter
}

func (h *WebSocketHandler) readPump(client *hub.Client) {
    client.limiter = rate.NewLimiter(rate.Every(100*time.Millisecond), 10)

    for {
        _, msgData, err := client.Conn.ReadMessage()
        if err != nil {
            break
        }

        // Check rate limit
        if !client.limiter.Allow() {
            log.Printf("Rate limit exceeded for client %s", client.ID)
            continue
        }

        // Process message
        // ...
    }
}
```

### JWT Best Practices

1. Use strong secret key (256-bit)
2. Set reasonable expiration (24 hours)
3. Implement token refresh
4. Validate all claims
5. Use HTTPS only

### XSS Protection

Sanitize message content:

```go
import "html"

func sanitizeContent(content string) string {
    return html.EscapeString(content)
}
```

---

## Troubleshooting

### Common Issues

#### 1. WebSocket Connection Fails

**Symptoms**: Connection refused, 404 errors

**Solutions**:
- Check if server is running: `curl http://localhost:8080/health`
- Verify WebSocket endpoint: `/ws`
- Check CORS configuration
- Validate query parameters (user_id, room_id)

#### 2. Messages Not Delivered

**Symptoms**: Messages sent but not received

**Solutions**:
- Check Redis connection: `redis-cli ping`
- Verify hub is running: Check logs for "Hub started"
- Ensure room_id matches between clients
- Check MongoDB for saved messages

#### 3. High Memory Usage

**Symptoms**: Memory usage grows over time

**Solutions**:
- Check for connection leaks: Monitor goroutine count
- Implement connection limits
- Add memory profiling: `go tool pprof`
- Review channel buffer sizes

#### 4. MongoDB Connection Errors

**Symptoms**: "connection refused", timeout errors

**Solutions**:
```bash
# Check MongoDB is running
docker ps | grep mongo

# Test connection
mongosh mongodb://localhost:27017

# Check logs
docker logs chatserver-mongo-1
```

#### 5. Redis Connection Issues

**Symptoms**: Pub/sub not working, messages not distributed

**Solutions**:
```bash
# Check Redis is running
docker ps | grep redis

# Test connection
redis-cli -h localhost -p 6379 ping

# Monitor pub/sub
redis-cli MONITOR
```

### Debugging Tips

1. **Enable Debug Logging**
   ```go
   log.SetFlags(log.LstdFlags | log.Lshortfile)
   ```

2. **Monitor Goroutines**
   ```go
   log.Printf("Active goroutines: %d", runtime.NumGoroutine())
   ```

3. **Track Connection Count**
   ```go
   log.Printf("Active connections: %d", hub.GetTotalClients())
   ```

4. **Use Request IDs**
   ```go
   requestID := uuid.New().String()
   log.Printf("[%s] Processing message", requestID)
   ```

### Performance Debugging

```bash
# CPU profiling
curl http://localhost:6060/debug/pprof/profile?seconds=30 > cpu.prof
go tool pprof cpu.prof

# Memory profiling
curl http://localhost:6060/debug/pprof/heap > mem.prof
go tool pprof mem.prof

# View goroutines
curl http://localhost:6060/debug/pprof/goroutine?debug=2
```

---

## API Reference

### WebSocket Endpoint

**URL**: `GET /ws`

**Query Parameters**:
- `user_id` (string, required): User identifier
- `room_id` (string, required): Room identifier

**Connection**:
```javascript
const ws = new WebSocket('ws://localhost:8080/ws?user_id=USER_ID&room_id=ROOM_ID');
```

### Message Format

**Client to Server**:
```json
{
  "type": "message",
  "room_id": "507f1f77bcf86cd799439011",
  "content": "Hello, World!"
}
```

**Server to Client**:
```json
{
  "id": "507f1f77bcf86cd799439011",
  "room_id": "507f1f77bcf86cd799439011",
  "user_id": "507f191e810c19729de860ea",
  "username": "john_doe",
  "content": "Hello, World!",
  "type": "text",
  "timestamp": "2024-01-15T10:30:00Z"
}
```

### Health Check

**URL**: `GET /health`

**Response**:
```
HTTP/1.1 200 OK
OK
```

---

## Additional Resources

### Documentation
- [Go WebSocket Guide](https://pkg.go.dev/github.com/gorilla/websocket)
- [MongoDB Go Driver](https://pkg.go.dev/go.mongodb.org/mongo-driver)
- [Redis Go Client](https://redis.uptrace.dev/)
- [Chi Router](https://go-chi.io/)

### Tools
- [Postman](https://www.postman.com/) - API testing
- [websocat](https://github.com/vi/websocat) - WebSocket CLI client
- [k6](https://k6.io/) - Load testing
- [MongoDB Compass](https://www.mongodb.com/products/compass) - MongoDB GUI

### Further Reading
- [Building Scalable WebSocket Applications](https://blog.logrocket.com/scalable-websockets-with-go-and-redis/)
- [Go Concurrency Patterns](https://go.dev/blog/pipelines)
- [Production-Ready Go Services](https://www.alexedwards.net/blog/organising-database-access)

---

## License

MIT License - See LICENSE file for details.

## Contributing

Contributions welcome! Please:
1. Fork the repository
2. Create a feature branch
3. Make your changes
4. Add tests
5. Submit a pull request

## Support

For issues or questions:
- Open an issue on GitHub
- Check existing issues for solutions
- Review troubleshooting section

---

**Last Updated**: January 2024
**Version**: 1.0.0
**Author**: Your Name
