package hub

import (
	"encoding/json"
	"log"
	"sync"

	"github.com/gorilla/websocket"
	"github.com/yourusername/chatserver/internal/models"
	"github.com/yourusername/chatserver/internal/pubsub"
)

type Client struct {
	ID     string
	UserID string
	RoomID string
	Conn   *websocket.Conn
	Send   chan []byte
}

type Hub struct {
	clients    map[string]*Client
	rooms      map[string]map[string]*Client
	broadcast  chan *models.Message
	register   chan *Client
	unregister chan *Client
	pubsub     *pubsub.RedisPubSub
	mu         sync.RWMutex
}

func New(ps *pubsub.RedisPubSub) *Hub {
	return &Hub{
		clients:    make(map[string]*Client),
		rooms:      make(map[string]map[string]*Client),
		broadcast:  make(chan *models.Message, 256),
		register:   make(chan *Client),
		unregister: make(chan *Client),
		pubsub:     ps,
	}
}

func (h *Hub) Run() {
	// Subscribe to Redis pub/sub
	h.pubsub.Subscribe("chat", func(message string) {
		var msg models.Message
		if err := json.Unmarshal([]byte(message), &msg); err != nil {
			log.Printf("Error unmarshaling message: %v", err)
			return
		}
		h.broadcastToRoom(msg.RoomID.Hex(), &msg)
	})

	for {
		select {
		case client := <-h.register:
			h.registerClient(client)

		case client := <-h.unregister:
			h.unregisterClient(client)

		case message := <-h.broadcast:
			// Publish to Redis for multi-server support
			data, err := json.Marshal(message)
			if err != nil {
				log.Printf("Error marshaling message: %v", err)
				continue
			}
			h.pubsub.Publish("chat", string(data))
		}
	}
}

func (h *Hub) registerClient(client *Client) {
	h.mu.Lock()
	defer h.mu.Unlock()

	h.clients[client.ID] = client

	if h.rooms[client.RoomID] == nil {
		h.rooms[client.RoomID] = make(map[string]*Client)
	}
	h.rooms[client.RoomID][client.ID] = client

	log.Printf("Client %s joined room %s", client.ID, client.RoomID)
}

func (h *Hub) unregisterClient(client *Client) {
	h.mu.Lock()
	defer h.mu.Unlock()

	if _, ok := h.clients[client.ID]; ok {
		delete(h.clients, client.ID)
		delete(h.rooms[client.RoomID], client.ID)
		close(client.Send)

		log.Printf("Client %s left room %s", client.ID, client.RoomID)
	}
}

func (h *Hub) broadcastToRoom(roomID string, message *models.Message) {
	h.mu.RLock()
	defer h.mu.RUnlock()

	clients, ok := h.rooms[roomID]
	if !ok {
		return
	}

	data, err := json.Marshal(message)
	if err != nil {
		log.Printf("Error marshaling message: %v", err)
		return
	}

	for _, client := range clients {
		select {
		case client.Send <- data:
		default:
			close(client.Send)
			delete(h.clients, client.ID)
			delete(h.rooms[roomID], client.ID)
		}
	}
}

func (h *Hub) Broadcast(message *models.Message) {
	h.broadcast <- message
}

func (h *Hub) GetRoomClients(roomID string) int {
	h.mu.RLock()
	defer h.mu.RUnlock()

	if clients, ok := h.rooms[roomID]; ok {
		return len(clients)
	}
	return 0
}

// Register registers a client with the hub
func (h *Hub) Register(client *Client) {
	h.register <- client
}

// Unregister unregisters a client from the hub
func (h *Hub) Unregister(client *Client) {
	h.unregister <- client
}
