# Unified REST API Platform with JWT Authentication

> A **production-ready** REST API built with Go, featuring JWT authentication, role-based access control (RBAC), rate limiting, and clean architecture principles. This comprehensive implementation demonstrates industry best practices for building secure, scalable APIs.

## Table of Contents

- [Quick Start](#quick-start)
- [Overview](#overview)
- [Features](#features)
- [Technology Stack](#technology-stack)
- [Architecture](#architecture)
- [Project Structure](#project-structure)
- [Installation & Setup](#installation--setup)
- [API Documentation](#api-documentation)
- [Authentication & Authorization](#authentication--authorization)
- [Security Features](#security-features)
- [Testing](#testing)
- [Deployment](#deployment)
- [Performance & Optimization](#performance--optimization)
- [Troubleshooting](#troubleshooting)
- [Contributing](#contributing)
- [License](#license)

---

## Quick Start

### Prerequisites
- Go 1.24+
- Git
- Optional: Docker & Docker Compose

### Get Running in 5 Minutes

```bash
# 1. Clone/extract the project
unzip restapi-solution.zip
cd restapi-solution

# 2. Install dependencies
go mod download
go mod tidy

# 3. Set environment variables (optional)
export JWT_SECRET=$(openssl rand -base64 32)
export PORT=8080

# 4. Run the application
go run cmd/api/main.go

# 5. Test the API
curl -X POST http://localhost:8080/api/v1/auth/register \
  -H "Content-Type: application/json" \
  -d '{
    "email": "user@example.com",
    "password": "SecurePass123!",
    "first_name": "John",
    "last_name": "Doe"
  }'
```

---

## Overview

This project is a **production-grade** REST API implementation that serves as a reference architecture for building secure, scalable APIs in Go. It demonstrates:

- **Clean Architecture** with clear separation of concerns
- **Enterprise-grade Security** with JWT and RBAC
- **Production Patterns** like rate limiting and graceful shutdown
- **Best Practices** throughout the codebase
- **Comprehensive Documentation** with examples and explanations

**Perfect for**:
- Learning Go API development
- Starting a new API project with best practices
- Understanding production patterns in Go
- Reference implementation for your own projects

---

## Features

### Core Features

✅ **JWT Authentication**
- Token-based authentication using industry-standard JWT
- Automatic token expiry (24 hours)
- Bearer token support
- Token validation and verification

✅ **Role-Based Access Control (RBAC)**
- Three-tier role system: Admin, User, Guest
- Fine-grained permission control
- Role-based route protection
- Permission matrix for different operations

✅ **User Management**
- User registration with validation
- Secure password hashing (bcrypt)
- User profile retrieval
- User listing (authenticated users only)

✅ **Security**
- Rate limiting (10 req/s, burst 20) - per IP
- Password complexity enforcement
- Protected routes with authentication
- Secure header configuration
- Password hashing with bcrypt (not plaintext)

✅ **Production Features**
- Graceful shutdown with timeout
- Request ID tracking
- Real IP detection
- Structured logging support
- Panic recovery middleware
- Configurable timeouts

### API Features

- **REST API** with semantic versioning (/api/v1)
- **Error Handling** with proper HTTP status codes
- **JSON Request/Response** with validation
- **Pagination Support** for list endpoints
- **CORS-Ready** architecture

---

## Technology Stack

| Component | Technology | Purpose | Version |
|-----------|-----------|---------|---------|
| **Language** | Go | Core application | 1.24+ |
| **HTTP Router** | Chi | Lightweight routing | v5 |
| **Authentication** | JWT | Token-based auth | golang-jwt/jwt v5 |
| **Password Hashing** | bcrypt | Secure passwords | golang.org/x/crypto |
| **Rate Limiting** | Token Bucket | Request throttling | golang.org/x/time |
| **Database** | PostgreSQL | Data persistence | 13+ (optional) |
| **Deployment** | Docker | Containerization | Latest |

**Why These Technologies?**
- **Go**: Fast, compiled, great for APIs, simple concurrency
- **Chi**: Minimal overhead, excellent middleware support, composable
- **JWT**: Stateless authentication, scalable, industry-standard
- **bcrypt**: Specifically designed for passwords, automatically salted
- **Token Bucket**: Proven rate limiting algorithm, fair to all clients

---

## Architecture

### System Architecture Diagram

```
┌─────────────────────────────────────────────────────────┐
│                    HTTP Clients                         │
│         (Browser, Mobile, Desktop, API)                 │
└────────────────────────┬────────────────────────────────┘
                         ↓
┌─────────────────────────────────────────────────────────┐
│                 HTTP Layer (Chi Router)                 │
│  ├─ Request routing by URL pattern                      │
│  ├─ Method-based routing (GET, POST, etc)               │
│  └─ Middleware chain composition                        │
└────────────────────────┬────────────────────────────────┘
                         ↓
┌─────────────────────────────────────────────────────────┐
│              Middleware Stack (Sequential)              │
│  1. Logger - Request/response logging                   │
│  2. Recoverer - Panic recovery                          │
│  3. RequestID - Unique request tracking                 │
│  4. RealIP - Extract real client IP                     │
│  5. RateLimit - Per-IP throttling                       │
│  6. Auth - JWT token validation                         │
│  7. RequireRole - RBAC enforcement                      │
└────────────────────────┬────────────────────────────────┘
                         ↓
┌─────────────────────────────────────────────────────────┐
│              Handler Layer (HTTP)                       │
│  ├─ Request parsing and validation                      │
│  ├─ Response formatting                                 │
│  └─ HTTP status code mapping                            │
└────────────────────────┬────────────────────────────────┘
                         ↓
┌─────────────────────────────────────────────────────────┐
│          Service Layer (Business Logic)                 │
│  ├─ Authentication logic                                │
│  ├─ User management                                     │
│  ├─ JWT operations                                      │
│  ├─ Password hashing/verification                       │
│  └─ Business rule enforcement                           │
└────────────────────────┬────────────────────────────────┘
                         ↓
┌─────────────────────────────────────────────────────────┐
│        Repository Layer (Data Access)                   │
│  ├─ Database operations                                 │
│  ├─ Data persistence                                    │
│  ├─ Transaction management                              │
│  └─ Query optimization                                  │
└────────────────────────┬────────────────────────────────┘
                         ↓
┌─────────────────────────────────────────────────────────┐
│            Database (PostgreSQL)                        │
│  ├─ User table                                          │
│  ├─ Indexed queries                                     │
│  └─ Transaction support                                 │
└─────────────────────────────────────────────────────────┘
```

### Request Flow Example

```
User Registration Request
│
├─ 1. HTTP POST /api/v1/auth/register
│      ├─ Email: user@example.com
│      ├─ Password: SecurePass123!
│      └─ Name: John Doe
│
├─ 2. Middleware Processing
│      ├─ Logger: Records request
│      ├─ RealIP: Identifies client IP
│      ├─ RateLimit: Checks request limit
│      └─ No auth required (public endpoint)
│
├─ 3. Handler Processing
│      ├─ Parse JSON body
│      ├─ Validate request format
│      └─ Call auth service
│
├─ 4. Service Layer (Business Logic)
│      ├─ Check if user exists (repository)
│      ├─ Hash password with bcrypt
│      ├─ Create user record (repository)
│      ├─ Generate JWT token
│      └─ Return AuthResponse
│
├─ 5. Repository Layer (Data Access)
│      ├─ Query database for existing user
│      ├─ Insert new user record
│      └─ Return ID and timestamps
│
├─ 6. Response Encoding
│      ├─ Serialize to JSON
│      ├─ Set HTTP 201 Created
│      └─ Include token and user data
│
└─ 7. Response
     {
       "token": "eyJhbGciOiJIUzI1NiIs...",
       "user": {
         "id": 1,
         "email": "user@example.com",
         "first_name": "John",
         "last_name": "Doe",
         "role": "user",
         "created_at": "2024-02-15T10:30:00Z"
       }
     }
```

### Authentication & Authorization Flow

```
LOGIN FLOW
┌─────────────────────────────────────┐
│ POST /api/v1/auth/login              │
│ {                                    │
│   "email": "user@example.com",       │
│   "password": "SecurePass123!"       │
│ }                                    │
└────────────┬────────────────────────┘
             ↓
┌─────────────────────────────────────┐
│ Service: Validate Credentials        │
│ 1. Fetch user by email               │
│ 2. Verify password with bcrypt       │
│ 3. Generate JWT token                │
└────────────┬────────────────────────┘
             ↓
┌─────────────────────────────────────┐
│ Return Token                         │
│ {                                    │
│   "token": "eyJhbGc...",             │
│   "user": {...}                      │
│ }                                    │
└─────────────────────────────────────┘


AUTHENTICATED REQUEST FLOW
┌─────────────────────────────────────┐
│ GET /api/v1/users                    │
│ Headers:                             │
│   Authorization: Bearer <token>      │
└────────────┬────────────────────────┘
             ↓
┌─────────────────────────────────────┐
│ Auth Middleware                      │
│ 1. Extract Authorization header      │
│ 2. Parse Bearer token                │
│ 3. Validate JWT signature            │
│ 4. Check expiration time             │
│ 5. Extract claims (user_id, role)    │
└────────────┬────────────────────────┘
             ↓
┌─────────────────────────────────────┐
│ Add to Context                       │
│ context.WithValue(                   │
│   ctx, UserContextKey, claims        │
│ )                                    │
└────────────┬────────────────────────┘
             ↓
┌─────────────────────────────────────┐
│ Handler Processing                   │
│ Retrieves user info from context     │
│ Processes request                    │
│ Returns data                         │
└─────────────────────────────────────┘


ROLE-BASED ACCESS CONTROL
┌─────────────────────────────────────┐
│ Authenticated Request                │
│ (Has token with claims)              │
└────────────┬────────────────────────┘
             ↓
┌─────────────────────────────────────┐
│ RequireRole Middleware               │
│ Check if claims.Role in allowed_roles│
└────────────┬────────────────────────┘
             ├─ Role matches? ─────── Yes ──> Proceed
             │
             └─ Role not allowed ──> 403 Forbidden
```

---

## Project Structure

```
restapi-solution/
├── cmd/
│   └── api/
│       └── main.go                  # Application entry point, router setup
│
├── internal/
│   ├── models/
│   │   └── user.go                  # Data models and DTOs
│   │
│   ├── repository/
│   │   └── user_repository.go       # Data access layer interface
│   │
│   ├── service/
│   │   ├── auth_service.go          # Authentication logic
│   │   └── user_service.go          # User business logic
│   │
│   └── api/
│       ├── handlers/
│       │   └── users.go             # HTTP request handlers
│       │
│       └── middleware/
│           ├── auth.go              # JWT authentication middleware
│           └── ratelimit.go         # Rate limiting middleware
│
├── migrations/                      # Database migration files
├── tests/
│   └── integration/                 # Integration test files
│
├── .gitignore                       # Git ignore patterns
├── go.mod                           # Go module definition
├── go.sum                           # Dependency checksums
└── README.md                        # This file
```

### Directory Conventions

| Directory | Purpose |
|-----------|---------|
| `cmd/` | Application entry points (main packages) |
| `internal/` | Private application code (not importable) |
| `internal/models/` | Data structures and DTOs |
| `internal/repository/` | Data access interfaces and implementations |
| `internal/service/` | Business logic layer |
| `internal/api/handlers/` | HTTP request handlers |
| `internal/api/middleware/` | HTTP middleware components |
| `migrations/` | Database schema migrations |
| `tests/` | Test files organized by type |

---

## Installation & Setup

### Prerequisites

**Required Software**:
- Go 1.24 or higher ([download](https://go.dev/dl/))
- PostgreSQL 13+ (optional for production database)
- Docker & Docker Compose (optional)

**Recommended Tools**:
- [Postman](https://www.postman.com/) or [Insomnia](https://insomnia.rest/) for API testing
- [TablePlus](https://tableplus.com/) or [DBeaver](https://dbeaver.io/) for database GUI
- [VS Code](https://code.visualstudio.com/) with Go extension

### Step 1: Extract and Navigate

```bash
# From ZIP file
unzip restapi-solution.zip
cd restapi-solution

# Or from Git
git clone https://github.com/yourusername/restapi.git
cd restapi
```

### Step 2: Install Dependencies

```bash
# Download dependencies
go mod download

# Tidy up dependencies
go mod tidy

# Verify installation
go mod verify
```

This installs:
- `github.com/go-chi/chi/v5` - HTTP router
- `github.com/golang-jwt/jwt/v5` - JWT library
- `golang.org/x/crypto` - bcrypt hashing
- `golang.org/x/time` - Rate limiting

### Step 3: Configure Environment

**Option A: Quick Start (Development)**

```bash
# No configuration needed - will use defaults
# PORT defaults to 8080
# JWT_SECRET defaults to "your-secret-key" (CHANGE IN PRODUCTION!)

go run cmd/api/main.go
```

**Option B: Custom Configuration**

Create `.env` file:

```bash
# Server
PORT=8080
ENV=development
LOG_LEVEL=info

# JWT
JWT_SECRET=your-super-secret-key-minimum-32-bytes-recommended
JWT_EXPIRY=24h

# Database (if using PostgreSQL)
DB_HOST=localhost
DB_PORT=5432
DB_NAME=restapi
DB_USER=postgres
DB_PASSWORD=postgres
DB_SSLMODE=disable

# Rate Limiting
RATE_LIMIT=10
RATE_BURST=20

# CORS (if needed)
CORS_ALLOWED_ORIGINS=http://localhost:3000,http://localhost:8080
```

Load environment in `main.go`:

```go
import "os"

func main() {
    port := os.Getenv("PORT")
    if port == "" {
        port = "8080"
    }
    // ... rest of setup
}
```

### Step 4: Run the Application

**Development Mode**:

```bash
go run cmd/api/main.go
```

**Production Build**:

```bash
# Build binary
go build -o bin/api cmd/api/main.go

# Run binary
./bin/api

# Optimized build (smaller size)
go build -ldflags="-s -w" -o bin/api cmd/api/main.go
```

**With Docker**:

```bash
# Build image
docker build -t restapi:latest .

# Run container
docker run -p 8080:8080 \
  -e JWT_SECRET="your-secret-key" \
  restapi:latest
```

### Step 5: Verify Installation

```bash
# Check if server is running
curl http://localhost:8080/api/v1/health

# Expected response: 200 OK

# Or register a test user
curl -X POST http://localhost:8080/api/v1/auth/register \
  -H "Content-Type: application/json" \
  -d '{
    "email": "test@example.com",
    "password": "TestPass123!",
    "first_name": "Test",
    "last_name": "User"
  }'
```

---

## API Documentation

### Base URL

```
http://localhost:8080/api/v1
```

### Authentication

All protected endpoints require a Bearer token in the Authorization header:

```http
Authorization: Bearer eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9...
```

### Authentication Endpoints

#### POST /auth/register

Register a new user account.

**Request**:
```http
POST /api/v1/auth/register HTTP/1.1
Content-Type: application/json

{
  "email": "john.doe@example.com",
  "password": "SecurePass123!",
  "first_name": "John",
  "last_name": "Doe"
}
```

**Validation Rules**:
- `email`: Valid email format, unique in database
- `password`: Minimum 8 characters
- `first_name`: Required, not empty
- `last_name`: Required, not empty

**Success Response** (201 Created):
```json
{
  "token": "eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJ1c2VyX2lkIjoxLCJlbWFpbCI6ImpvaG4uZG9lQGV4YW1wbGUuY29tIiwicm9sZSI6InVzZXIiLCJleHAiOjE3MDg1NjAwMDAsImlhdCI6MTcwODQ3MzYwMH0.SflKxwRJSMeKKF2QT4fwpMeJf36POk6yJV_adQssw5c",
  "user": {
    "id": 1,
    "email": "john.doe@example.com",
    "first_name": "John",
    "last_name": "Doe",
    "role": "user",
    "created_at": "2024-02-15T10:30:00Z",
    "updated_at": "2024-02-15T10:30:00Z"
  }
}
```

**Error Response** (400 Bad Request):
```json
{
  "error": "validation failed",
  "details": {
    "email": "invalid email format",
    "password": "password must be at least 8 characters"
  }
}
```

**Error Response** (409 Conflict - Email exists):
```json
{
  "error": "user already exists"
}
```

---

#### POST /auth/login

Authenticate user and receive JWT token.

**Request**:
```http
POST /api/v1/auth/login HTTP/1.1
Content-Type: application/json

{
  "email": "john.doe@example.com",
  "password": "SecurePass123!"
}
```

**Success Response** (200 OK):
```json
{
  "token": "eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9...",
  "user": {
    "id": 1,
    "email": "john.doe@example.com",
    "first_name": "John",
    "last_name": "Doe",
    "role": "user",
    "created_at": "2024-02-15T10:30:00Z",
    "updated_at": "2024-02-15T10:30:00Z"
  }
}
```

**Error Response** (401 Unauthorized):
```json
{
  "error": "invalid credentials"
}
```

---

### User Endpoints

#### GET /users

List all users (requires authentication).

**Request**:
```http
GET /api/v1/users HTTP/1.1
Authorization: Bearer eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9...
```

**Query Parameters**:
- `page` (optional): Page number, default 1
- `limit` (optional): Results per page, default 20, max 100

**Success Response** (200 OK):
```json
{
  "users": [
    {
      "id": 1,
      "email": "john.doe@example.com",
      "first_name": "John",
      "last_name": "Doe",
      "role": "user",
      "created_at": "2024-02-15T10:30:00Z",
      "updated_at": "2024-02-15T10:30:00Z"
    },
    {
      "id": 2,
      "email": "jane.smith@example.com",
      "first_name": "Jane",
      "last_name": "Smith",
      "role": "admin",
      "created_at": "2024-02-14T09:15:00Z",
      "updated_at": "2024-02-14T09:15:00Z"
    }
  ],
  "pagination": {
    "page": 1,
    "limit": 20,
    "total": 2,
    "pages": 1
  }
}
```

**Error Response** (401 Unauthorized - No token):
```json
{
  "error": "unauthorized"
}
```

---

#### GET /users/{id}

Get user by ID (requires authentication).

**Request**:
```http
GET /api/v1/users/1 HTTP/1.1
Authorization: Bearer eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9...
```

**Success Response** (200 OK):
```json
{
  "id": 1,
  "email": "john.doe@example.com",
  "first_name": "John",
  "last_name": "Doe",
  "role": "user",
  "created_at": "2024-02-15T10:30:00Z",
  "updated_at": "2024-02-15T10:30:00Z"
}
```

**Error Response** (404 Not Found):
```json
{
  "error": "user not found"
}
```

---

### HTTP Status Codes

| Code | Status | When Used |
|------|--------|-----------|
| 200 | OK | Successful GET/PUT request |
| 201 | Created | Successful POST request |
| 400 | Bad Request | Invalid input or validation failure |
| 401 | Unauthorized | Missing or invalid authentication token |
| 403 | Forbidden | Insufficient permissions for action |
| 404 | Not Found | Resource doesn't exist |
| 409 | Conflict | Resource already exists (duplicate email) |
| 429 | Too Many Requests | Rate limit exceeded |
| 500 | Internal Server Error | Server error |

---

## Authentication & Authorization

### JWT Token Structure

**Generated Token Example**:
```
eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJ1c2VyX2lkIjoxLCJlbWFpbCI6InVzZXJAZXhhbXBsZS5jb20iLCJyb2xlIjoidXNlciIsImV4cCI6MTcwODU2MDAwMCwiaWF0IjoxNzA4NDczNjAwfQ.SflKxwRJSMeKKF2QT4fwpMeJf36POk6yJV_adQssw5c
```

**Token Components** (separated by dots):

1. **Header** (base64 decoded):
```json
{
  "alg": "HS256",
  "typ": "JWT"
}
```

2. **Payload** (base64 decoded):
```json
{
  "user_id": 1,
  "email": "user@example.com",
  "role": "user",
  "exp": 1708560000,
  "iat": 1708473600
}
```

3. **Signature**: HMAC-SHA256(header + payload, secret_key)

**Claims Explained**:
- `user_id`: Unique user identifier
- `email`: User's email address
- `role`: User's role (admin, user, guest)
- `exp`: Expiration time (Unix timestamp)
- `iat`: Issued at time (Unix timestamp)

### Password Hashing

**Bcrypt Process**:
```
Plain Password: "MyPassword123!"
                    ↓
         Generate Random Salt
                    ↓
         Apply Bcrypt Hash
         (Cost Factor: 10)
                    ↓
Stored Hash: "$2a$10$N9qo8uLOickgx2ZMRZoMyeIjZAgcfl7p92ldGxad68LJZdL17lhWy"
```

**Why Bcrypt?**
- Designed specifically for passwords (slow by design)
- Automatically includes salt (prevents rainbow tables)
- Configurable cost factor (increases difficulty over time)
- Resistant to GPU attacks (computationally expensive)

**Password Verification**:
```go
// During login
err := bcrypt.CompareHashAndPassword(
    []byte(storedHash),     // From database
    []byte(userPassword),   // From request
)
// Returns nil if passwords match
```

### Role-Based Access Control (RBAC)

**Role Hierarchy**:
```
┌─────────────┐
│   Admin     │  Full system access
└──────┬──────┘
       │ Can do everything User can do
┌──────▼───────┐
│   User       │  Standard access
└──────┬───────┘
       │ Can do everything Guest can do
┌──────▼───────┐
│   Guest      │  Read-only access
└──────────────┘
```

**Permission Matrix**:

| Action | Guest | User | Admin |
|--------|-------|------|-------|
| Register | ✅ Yes | ✅ Yes | ✅ Yes |
| Login | ✅ Yes | ✅ Yes | ✅ Yes |
| View own profile | ❌ No | ✅ Yes | ✅ Yes |
| View all users | ❌ No | ✅ Yes | ✅ Yes |
| Update own profile | ❌ No | ✅ Yes | ✅ Yes |
| Update any profile | ❌ No | ❌ No | ✅ Yes |
| Delete users | ❌ No | ❌ No | ✅ Yes |
| Manage roles | ❌ No | ❌ No | ✅ Yes |

**Implementation in Code**:

```go
// Public routes (anyone)
r.Post("/auth/register", authHandler.Register)
r.Post("/auth/login", authHandler.Login)

// Protected routes (authenticated users)
r.Group(func(r chi.Router) {
    r.Use(apiMiddleware.Auth(authService))

    // Any authenticated user
    r.Get("/users", userHandler.ListUsers)
    r.Get("/users/{id}", userHandler.GetUser)

    // Admin only
    r.Group(func(r chi.Router) {
        r.Use(apiMiddleware.RequireRole(models.RoleAdmin))

        r.Delete("/users/{id}", userHandler.DeleteUser)
        r.Put("/users/{id}/role", userHandler.UpdateUserRole)
    })
})
```

---

## Security Features

### 1. Rate Limiting

**Token Bucket Algorithm**:
```
Bucket capacity: 20 tokens (burst)
Refill rate: 10 tokens/second

Request arrives:
  - If tokens available: consume 1 token, allow request
  - If no tokens: reject with 429 Too Many Requests

Time passes:
  - Tokens refill at rate (capped at capacity)
```

**Current Configuration**:
- Limit: 10 requests/second per IP
- Burst: 20 requests allowed at once
- Applies to all routes

**Customization**:
```go
// Strict limit for auth endpoints
authLimiter := NewRateLimiter(rate.Limit(5), 10)

// Lenient limit for general API
apiLimiter := NewRateLimiter(rate.Limit(100), 200)
```

### 2. Password Security

**Requirements**:
- Minimum 8 characters
- Can be enhanced with:
  - Uppercase letters
  - Lowercase letters
  - Numbers
  - Special characters

**Hashing**:
- Uses bcrypt with cost factor 10
- ~100ms to hash a password (secure but not slow)
- Automatically generates and includes salt

### 3. JWT Token Security

**Secret Management**:

```bash
# Generate secure JWT secret (32 bytes / 256 bits)
openssl rand -base64 32
```

**Configuration**:

```go
// DO NOT hardcode in code
var jwtSecret = []byte(os.Getenv("JWT_SECRET"))

// In production, use secret management service
jwtSecret := getSecretFromVault("jwt-secret")
```

**Requirements**:
- Minimum 256 bits (32 bytes)
- Cryptographically random
- Unique per environment
- Rotate every 90 days

### 4. Input Validation

**Request Validation**:
- Email format validation
- Password minimum length
- Required field checks
- Type validation

**Implementation**:
```go
type RegisterRequest struct {
    Email     string `json:"email" validate:"required,email"`
    Password  string `json:"password" validate:"required,min=8"`
    FirstName string `json:"first_name" validate:"required"`
    LastName  string `json:"last_name" validate:"required"`
}
```

### 5. HTTP Security Headers

**Recommended Headers** (add to middleware):

```go
func SecurityHeaders(next http.Handler) http.Handler {
    return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
        w.Header().Set("X-Content-Type-Options", "nosniff")
        w.Header().Set("X-Frame-Options", "DENY")
        w.Header().Set("X-XSS-Protection", "1; mode=block")
        w.Header().Set("Strict-Transport-Security", "max-age=31536000")
        next.ServeHTTP(w, r)
    })
}

r.Use(SecurityHeaders)
```

### 6. HTTPS/TLS Configuration

**Production HTTPS**:

```go
srv := &http.Server{
    Addr:    ":443",
    Handler: r,
    TLSConfig: &tls.Config{
        MinVersion: tls.VersionTLS13,
        CipherSuites: []uint16{
            tls.TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384,
            tls.TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384,
        },
    },
}

log.Fatal(srv.ListenAndServeTLS("cert.pem", "key.pem"))
```

**Get SSL Certificate**:

```bash
# Using Let's Encrypt (certbot)
sudo certbot certonly --standalone -d yourdomain.com
```

---

## Testing

### Running Tests

```bash
# Run all tests
go test ./...

# Run with verbose output
go test -v ./...

# Run specific test
go test -v -run TestRegister ./internal/service

# Run with coverage
go test -cover ./...

# Generate HTML coverage report
go test -coverprofile=coverage.out ./...
go tool cover -html=coverage.out -o coverage.html
```

### Testing Examples

**Unit Test Example**:

```bash
curl -X POST http://localhost:8080/api/v1/auth/register \
  -H "Content-Type: application/json" \
  -d '{
    "email": "test1@example.com",
    "password": "TestPass123!",
    "first_name": "Test",
    "last_name": "User"
  }'
```

**Authenticated Request Example**:

```bash
# 1. Register and get token
TOKEN=$(curl -s -X POST http://localhost:8080/api/v1/auth/register \
  -H "Content-Type: application/json" \
  -d '{
    "email": "test2@example.com",
    "password": "TestPass123!",
    "first_name": "Test",
    "last_name": "User"
  }' | jq -r '.token')

# 2. Use token in authenticated request
curl -X GET http://localhost:8080/api/v1/users \
  -H "Authorization: Bearer $TOKEN"
```

**Rate Limiting Test**:

```bash
# Send requests rapidly - should get 429 after limit
for i in {1..25}; do
  echo "Request $i:"
  curl -s -w "HTTP %{http_code}\n" http://localhost:8080/api/v1/users \
    -H "Authorization: Bearer $TOKEN"
  sleep 0.05
done
```

---

## Deployment

### Docker Deployment

**Dockerfile**:
```dockerfile
# Build stage
FROM golang:1.24-alpine AS builder
WORKDIR /app
COPY go.mod go.sum ./
RUN go mod download
COPY . .
RUN CGO_ENABLED=0 GOOS=linux go build -o main ./cmd/api

# Runtime stage
FROM alpine:latest
RUN apk --no-cache add ca-certificates
COPY --from=builder /app/main .
EXPOSE 8080
CMD ["./main"]
```

**Build and Run**:
```bash
# Build image
docker build -t restapi:latest .

# Run container
docker run -p 8080:8080 \
  -e JWT_SECRET="your-secret-key" \
  restapi:latest

# Run with environment file
docker run -p 8080:8080 \
  --env-file .env \
  restapi:latest
```

### Docker Compose Deployment

**docker-compose.yml**:
```yaml
version: '3.8'

services:
  api:
    build: .
    ports:
      - "8080:8080"
    environment:
      - JWT_SECRET=${JWT_SECRET}
      - DB_HOST=db
      - DB_NAME=restapi
    depends_on:
      - db

  db:
    image: postgres:15-alpine
    environment:
      - POSTGRES_DB=restapi
      - POSTGRES_PASSWORD=${DB_PASSWORD}
    volumes:
      - postgres_data:/var/lib/postgresql/data

volumes:
  postgres_data:
```

**Deploy**:
```bash
# Start services
docker-compose up -d

# View logs
docker-compose logs -f api

# Stop services
docker-compose down
```

### Environment Variables for Production

```bash
# Server
export PORT=8080
export ENV=production
export LOG_LEVEL=info

# JWT
export JWT_SECRET=$(openssl rand -base64 32)
export JWT_EXPIRY=24h

# Database
export DB_HOST=your-db-host.example.com
export DB_PORT=5432
export DB_NAME=restapi
export DB_USER=restapi_user
export DB_PASSWORD=secure-password
export DB_SSLMODE=require

# Connection Pool
export DB_MAX_OPEN_CONNS=25
export DB_MAX_IDLE_CONNS=5
export DB_CONN_MAX_LIFETIME=5m
```

---

## Performance & Optimization

### Database Connection Pooling

```go
db.SetMaxOpenConns(25)                // Maximum open connections
db.SetMaxIdleConns(5)                 // Maximum idle connections
db.SetConnMaxLifetime(5 * time.Minute) // Connection lifetime
```

### Response Caching

```go
// Cache GET requests for 5 minutes
r.With(CacheMiddleware(5*time.Minute)).Get("/api/v1/users", ...)
```

### Profiling Performance

```bash
# CPU profiling
go test -cpuprofile=cpu.prof -bench=.
go tool pprof cpu.prof

# Memory profiling
go test -memprofile=mem.prof -bench=.
go tool pprof mem.prof

# Web UI
go tool pprof -http=:8081 cpu.prof
```

---

## Troubleshooting

### Common Issues and Solutions

#### "Cannot start server on port 8080"

```bash
# Check if port is in use
lsof -i :8080

# Kill process on port
kill -9 <PID>

# Use different port
PORT=3000 go run cmd/api/main.go
```

#### "Invalid token" errors

```bash
# Check JWT_SECRET is set
echo $JWT_SECRET

# Ensure it's at least 32 bytes
echo -n "your-secret" | wc -c

# Generate new secret if needed
openssl rand -base64 32
```

#### "Rate limit exceeded" in development

```go
// Increase limits for development
if os.Getenv("ENV") == "development" {
    rateLimiter = NewRateLimiter(rate.Limit(1000), 2000)
}
```

### Enable Debug Logging

```go
logger := zap.NewDevelopment()
logger.Debug("Request details",
    zap.String("method", r.Method),
    zap.String("path", r.URL.Path),
)
```

---

## Contributing

We welcome contributions! Please follow these guidelines:

1. **Fork** the repository
2. **Create** a feature branch (`git checkout -b feature/amazing-feature`)
3. **Write** tests for new functionality
4. **Commit** changes with clear messages
5. **Push** to your branch
6. **Open** a Pull Request

**Code Standards**:
- Run `go fmt` before committing
- Add tests for new features
- Update documentation
- Follow Go best practices

---

## License

This project is licensed under the MIT License - see the LICENSE file for details.

---

## Additional Resources

**Learning Materials**:
- [Go by Example](https://gobyexample.com/)
- [Effective Go](https://go.dev/doc/effective_go)
- [JWT.io](https://jwt.io/) - JWT debugger and info
- [Go Web Examples](https://gowebexamples.com/)
- [Chi Router Documentation](https://go-chi.io/)

**Tools**:
- [Postman](https://www.postman.com/) - API testing
- [Insomnia](https://insomnia.rest/) - API client
- [JWT Debugger](https://jwt.io/) - Decode and verify JWTs

---

**Last Updated**: 2024-10-29
**Version**: 2.0.0 - Enhanced Documentation
**Maintained by**: Go Tutorial Team

For issues, questions, or suggestions, please open an issue or submit a pull request on GitHub.
