package main

import (
	"context"
	"log"
	"net/http"
	"os"
	"os/signal"
	"syscall"
	"time"

	"github.com/go-chi/chi/v5"
	"github.com/go-chi/chi/v5/middleware"
	_ "github.com/yourusername/restapi/internal/api/handlers"
	apiMiddleware "github.com/yourusername/restapi/internal/api/middleware"
	"github.com/yourusername/restapi/internal/models"
	_ "github.com/yourusername/restapi/internal/repository"
	_ "github.com/yourusername/restapi/internal/service"
	"golang.org/x/time/rate"
)

// @title REST API with Authentication
// @version 1.0
// @description Production-ready REST API with JWT authentication
// @host localhost:8080
// @BasePath /api/v1
// @securityDefinitions.apikey BearerAuth
// @in header
// @name Authorization
func main() {
	// Database connection (simplified)
	// db := initDB()

	// Initialize repositories
	// userRepo := repository.NewUserRepository(db)

	// Initialize services
	// authService := service.NewAuthService(userRepo)
	// userService := service.NewUserService(userRepo)

	// Initialize handlers
	// authHandler := handlers.NewAuthHandler(authService)
	// userHandler := handlers.NewUserHandler(userService)

	// Setup router
	r := chi.NewRouter()

	// Global middleware
	r.Use(middleware.Logger)
	r.Use(middleware.Recoverer)
	r.Use(middleware.RequestID)
	r.Use(middleware.RealIP)

	// Rate limiting
	rateLimiter := apiMiddleware.NewRateLimiter(rate.Limit(10), 20)
	r.Use(rateLimiter.Middleware)

	// Public routes
	r.Route("/api/v1", func(r chi.Router) {
		// Uncomment when authHandler is initialized
		// r.Post("/auth/register", authHandler.Register)
		// r.Post("/auth/login", authHandler.Login)

		// Protected routes
		r.Group(func(r chi.Router) {
			// Uncomment when authService is initialized
			// r.Use(apiMiddleware.Auth(authService))

			// User routes
			// r.Get("/users", userHandler.ListUsers)
			// r.Get("/users/{id}", userHandler.GetUser)

			// Admin only routes
			r.Group(func(r chi.Router) {
				r.Use(apiMiddleware.RequireRole(models.RoleAdmin))
				// r.Delete("/users/{id}", userHandler.DeleteUser)
			})
		})
	})

	// Start server with graceful shutdown
	srv := &http.Server{
		Addr:         ":8080",
		Handler:      r,
		ReadTimeout:  15 * time.Second,
		WriteTimeout: 15 * time.Second,
		IdleTimeout:  60 * time.Second,
	}

	go func() {
		log.Println("Server starting on :8080")
		if err := srv.ListenAndServe(); err != nil && err != http.ErrServerClosed {
			log.Fatalf("Server error: %v", err)
		}
	}()

	quit := make(chan os.Signal, 1)
	signal.Notify(quit, syscall.SIGINT, syscall.SIGTERM)
	<-quit

	ctx, cancel := context.WithTimeout(context.Background(), 30*time.Second)
	defer cancel()

	if err := srv.Shutdown(ctx); err != nil {
		log.Fatal("Server forced to shutdown:", err)
	}

	log.Println("Server exited gracefully")
}
