# Quick Start Examples

## Example 1: Single Node (Testing)

Start a single node for quick testing:

```bash
# Build
go build -o kvstore ./cmd/kvstore

# Run
./kvstore --node-id=node1 --bootstrap
```

Test it:
```bash
# Set a value
curl -X POST http://localhost:8080/api/set \
  -H "Content-Type: application/json" \
  -d '{"key":"test","value":"hello"}'

# Get the value
curl http://localhost:8080/api/get?key=test

# Result: {"value":"hello","found":true}
```

## Example 2: Three Node Cluster (Recommended)

### Terminal 1 - Leader Node
```bash
./kvstore \
  --node-id=node1 \
  --raft-addr=localhost:7000 \
  --http-addr=:8081 \
  --data-dir=./data/node1 \
  --bootstrap
```

### Terminal 2 - Follower Node 1
```bash
./kvstore \
  --node-id=node2 \
  --raft-addr=localhost:7001 \
  --http-addr=:8082 \
  --data-dir=./data/node2
```

### Terminal 3 - Follower Node 2
```bash
./kvstore \
  --node-id=node3 \
  --raft-addr=localhost:7002 \
  --http-addr=:8083 \
  --data-dir=./data/node3
```

### Terminal 4 - Join Cluster
```bash
# Add node2
curl -X POST http://localhost:8081/api/join \
  -H "Content-Type: application/json" \
  -d '{"node_id":"node2","addr":"localhost:7001"}'

# Add node3
curl -X POST http://localhost:8081/api/join \
  -H "Content-Type: application/json" \
  -d '{"node_id":"node3","addr":"localhost:7002"}'
```

## Example 3: Complete Workflow

```bash
# 1. Write data to leader
curl -X POST http://localhost:8081/api/set \
  -H "Content-Type: application/json" \
  -d '{"key":"user:alice","value":"Engineer"}'

curl -X POST http://localhost:8081/api/set \
  -H "Content-Type: application/json" \
  -d '{"key":"user:bob","value":"Designer"}'

# 2. Read from any node (data is replicated)
curl http://localhost:8081/api/get?key=user:alice  # From leader
curl http://localhost:8082/api/get?key=user:alice  # From follower 1
curl http://localhost:8083/api/get?key=user:alice  # From follower 2

# 3. List keys with prefix
curl http://localhost:8081/api/list?prefix=user:

# Result:
# {
#   "pairs": {
#     "user:alice": "Engineer",
#     "user:bob": "Designer"
#   }
# }

# 4. Delete a key
curl -X DELETE http://localhost:8081/api/delete?key=user:bob

# 5. Verify deletion
curl http://localhost:8081/api/get?key=user:bob
# Result: {"found":false}
```

## Example 4: Monitoring

```bash
# Check cluster stats
curl http://localhost:8081/api/stats | jq '.'

# Example output:
# {
#   "stats": {
#     "state": "Leader",
#     "term": "2",
#     "last_log_index": "7",
#     "commit_index": "7",
#     ...
#   },
#   "is_leader": true,
#   "leader": "127.0.0.1:7000"
# }

# Health check
curl http://localhost:8081/health | jq '.'

# Example output:
# {
#   "status": "ok",
#   "node_id": "node1",
#   "is_leader": true,
#   "leader": "127.0.0.1:7000"
# }
```

## Example 5: Fault Tolerance Demo

```bash
# 1. Start 3-node cluster (see Example 2)

# 2. Write data
curl -X POST http://localhost:8081/api/set \
  -H "Content-Type: application/json" \
  -d '{"key":"important","value":"data"}'

# 3. Kill one follower (Ctrl+C in Terminal 3)

# 4. Verify cluster still works (2 out of 3 is quorum)
curl -X POST http://localhost:8081/api/set \
  -H "Content-Type: application/json" \
  -d '{"key":"resilient","value":"system"}'

# Success! Cluster tolerates 1 failure

# 5. Read from remaining nodes
curl http://localhost:8081/api/get?key=resilient  # Works
curl http://localhost:8082/api/get?key=resilient  # Works
```

## Example 6: Using Environment Variables

```bash
# Set environment variables
export NODE_ID=mynode
export RAFT_ADDR=localhost:7000
export HTTP_ADDR=:8080
export DATA_DIR=./mydata
export BOOTSTRAP=true

# Run with env vars
./kvstore
```

## Example 7: Automated Demo

Use the included demo script:

```bash
# Make it executable
chmod +x demo.sh

# Run complete demo
./demo.sh
```

This will:
- Build the project
- Start a 3-node cluster
- Join nodes together
- Perform various operations
- Test fault tolerance
- Leave cluster running for you to experiment

## Common Operations

### List All Keys
```bash
curl http://localhost:8081/api/list?prefix=
```

### Batch Insert
```bash
for i in {1..10}; do
  curl -X POST http://localhost:8081/api/set \
    -H "Content-Type: application/json" \
    -d "{\"key\":\"item:$i\",\"value\":\"value$i\"}"
done
```

### Check if Key Exists
```bash
curl http://localhost:8081/api/get?key=mykey | jq '.found'
```

### Count Keys with Prefix
```bash
curl -s http://localhost:8081/api/list?prefix=user: | jq '.pairs | length'
```

## Troubleshooting

### Cannot Connect to Node
```bash
# Check if node is running
curl http://localhost:8081/health

# Check logs (if running in background)
tail -f ./data/node1.log
```

### Write Fails with "not leader"
```bash
# Find the leader
curl http://localhost:8081/api/stats | jq '.leader'

# Write to leader
curl -X POST http://LEADER_ADDR/api/set ...
```

### Clean Restart
```bash
# Stop all nodes
pkill kvstore

# Remove data
rm -rf ./data

# Restart from scratch
```

## Next Steps

After trying these examples:

1. Modify the code to add new features
2. Add persistent storage (replace memory.go with disk-based storage)
3. Implement authentication
4. Add metrics and monitoring
5. Create a CLI client
6. Experiment with different cluster sizes

Happy learning!
