# Distributed KV Store - Quick Start Guide

Get up and running with the distributed KV store in 5 minutes.

## Table of Contents

1. [Requirements](#requirements)
2. [Single Node Setup (2 min)](#single-node-setup)
3. [3-Node Cluster (3 min)](#3-node-cluster)
4. [Basic Operations](#basic-operations)
5. [Next Steps](#next-steps)

---

## Requirements

- Go 1.24+ (or Docker)
- curl for API testing
- 8 GB disk space for data

---

## Single Node Setup

### Step 1: Build the Binary

```bash
git clone <repo-url>
cd kvstore-simplified
go build -o kvstore ./cmd/kvstore
```

### Step 2: Start the Node

```bash
./kvstore --bootstrap
```

You should see:
```
2025/10/22 10:30:15 === Starting Distributed KV Store ===
2025/10/22 10:30:15 Node ID: node1
2025/10/22 10:30:15 HTTP address: :8080
2025/10/22 10:30:16 Node ready to serve requests
```

### Step 3: Test It

```bash
# Set a value
curl -X POST http://localhost:8080/api/set \
  -H "Content-Type: application/json" \
  -d '{"key":"hello","value":"world"}'

# Get the value
curl http://localhost:8080/api/get?key=hello

# Expected output:
# {"value":"world","found":true}
```

**Congratulations!** Your single-node KV store is running.

---

## 3-Node Cluster

### Setup Instructions

Open 4 terminals:

**Terminal 1 - Start Node 1 (Leader)**
```bash
./kvstore \
  --node-id=node1 \
  --raft-addr=localhost:7000 \
  --http-addr=:8081 \
  --data-dir=./data/node1 \
  --bootstrap
```

Wait for: `Node ready to serve requests`

**Terminal 2 - Start Node 2**
```bash
./kvstore \
  --node-id=node2 \
  --raft-addr=localhost:7001 \
  --http-addr=:8082 \
  --data-dir=./data/node2
```

**Terminal 3 - Start Node 3**
```bash
./kvstore \
  --node-id=node3 \
  --raft-addr=localhost:7002 \
  --http-addr=:8083 \
  --data-dir=./data/node3
```

**Terminal 4 - Join the Cluster**
```bash
# Give nodes a moment to start
sleep 3

# Join node2
curl -X POST http://localhost:8081/api/join \
  -H "Content-Type: application/json" \
  -d '{"node_id":"node2","addr":"localhost:7001"}'

sleep 1

# Join node3
curl -X POST http://localhost:8081/api/join \
  -H "Content-Type: application/json" \
  -d '{"node_id":"node3","addr":"localhost:7002"}'

# Verify cluster
curl http://localhost:8081/api/stats | jq '.is_leader, .leader'
```

---

## Basic Operations

### Write Data (to any node, but should reach leader)

```bash
# Set single value
curl -X POST http://localhost:8081/api/set \
  -H "Content-Type: application/json" \
  -d '{"key":"user:alice","value":"alice@example.com"}'

# Set another value
curl -X POST http://localhost:8082/api/set \
  -H "Content-Type: application/json" \
  -d '{"key":"user:bob","value":"bob@example.com"}'
```

### Read Data (from any node)

```bash
# Read from node 1
curl http://localhost:8081/api/get?key=user:alice

# Read from node 2 (should have same data)
curl http://localhost:8082/api/get?key=user:alice

# Read from node 3 (should have same data)
curl http://localhost:8083/api/get?key=user:alice
```

### List Keys (with prefix)

```bash
# List all keys starting with "user:"
curl http://localhost:8081/api/list?prefix=user:

# Expected output:
# {
#   "pairs": {
#     "user:alice": "alice@example.com",
#     "user:bob": "bob@example.com"
#   }
# }
```

### Delete Data

```bash
curl -X DELETE http://localhost:8081/api/delete?key=user:bob

# Verify deletion
curl http://localhost:8081/api/get?key=user:bob
# {"found":false}
```

### Check Cluster Health

```bash
# Get stats
curl http://localhost:8081/api/stats | jq '.'

# Health check
curl http://localhost:8081/health | jq '.'

# Expected output (leader):
# {
#   "status": "ok",
#   "node_id": "node1",
#   "is_leader": true,
#   "leader": "127.0.0.1:7000"
# }
```

---

## Common Patterns

### Batch Write Multiple Keys

```bash
#!/bin/bash
for i in {1..10}; do
  curl -X POST http://localhost:8081/api/set \
    -H "Content-Type: application/json" \
    -d "{\"key\":\"item:$i\",\"value\":\"Item $i\"}"
done

# Verify
curl http://localhost:8081/api/list?prefix=item: | jq '.pairs | length'
```

### Test Fault Tolerance

```bash
# 1. Kill one follower (Ctrl+C in Terminal 3)
# Kill node3

# 2. Cluster still works with 2 out of 3 nodes
curl -X POST http://localhost:8081/api/set \
  -H "Content-Type: application/json" \
  -d '{"key":"resilient","value":"yes"}'

# 3. Verify data available
curl http://localhost:8081/api/get?key=resilient
curl http://localhost:8082/api/get?key=resilient

# 4. Restart node3
./kvstore \
  --node-id=node3 \
  --raft-addr=localhost:7002 \
  --http-addr=:8083 \
  --data-dir=./data/node3

# 5. Rejoin to cluster
curl -X POST http://localhost:8081/api/join \
  -H "Content-Type: application/json" \
  -d '{"node_id":"node3","addr":"localhost:7002"}'

# 6. Verify node3 caught up
curl http://localhost:8083/api/get?key=resilient
```

---

## Using Docker

### Build Image

```bash
docker build -t kvstore:latest .
```

### Run Single Node

```bash
docker run -d \
  --name kvstore-single \
  -p 8080:8080 \
  -p 7000:7000 \
  -v kvstore-data:/home/appuser/data \
  kvstore:latest
```

### Run 3-Node Cluster with Docker Compose

```bash
# Start cluster
docker-compose up -d

# Check status
docker-compose ps
docker-compose logs -f

# Test
curl -X POST http://localhost:8081/api/set \
  -H "Content-Type: application/json" \
  -d '{"key":"docker","value":"compose"}'

curl http://localhost:8082/api/get?key=docker

# Stop
docker-compose down
```

---

## Using Demo Script

```bash
# Make script executable
chmod +x demo.sh

# Run automated demo
./demo.sh

# The script will:
# 1. Build the binary
# 2. Start 3-node cluster
# 3. Join nodes automatically
# 4. Perform sample operations
# 5. Test fault tolerance
# 6. Leave cluster running for you to experiment
```

---

## API Quick Reference

| Method | Endpoint | Purpose | Example |
|--------|----------|---------|---------|
| POST | /api/set | Write value | `curl -X POST http://host:8080/api/set -d '{"key":"k","value":"v"}'` |
| GET | /api/get | Read value | `curl http://host:8080/api/get?key=k` |
| DELETE | /api/delete | Delete key | `curl -X DELETE http://host:8080/api/delete?key=k` |
| GET | /api/list | List keys | `curl http://host:8080/api/list?prefix=user:` |
| POST | /api/join | Join cluster | `curl -X POST http://host:8080/api/join -d '{"node_id":"n2","addr":"host:7001"}'` |
| GET | /api/stats | Get stats | `curl http://host:8080/api/stats` |
| GET | /health | Health check | `curl http://host:8080/health` |

---

## Troubleshooting

### Issue: "Address already in use"

```bash
# Kill existing process
pkill kvstore

# Or use different ports
./kvstore --http-addr=:9000 --raft-addr=localhost:8000
```

### Issue: "not leader" error on write

```bash
# Find the leader
LEADER=$(curl -s http://localhost:8081/api/stats | jq -r '.leader')

# Write to leader instead
curl -X POST http://$LEADER/api/set \
  -H "Content-Type: application/json" \
  -d '{"key":"test","value":"value"}'
```

### Issue: Nodes can't communicate

```bash
# Check connectivity
nc -zv localhost 7000

# Check firewall
sudo ufw status

# Allow ports if needed
sudo ufw allow 7000:7010/tcp
```

### Issue: Data not replicating

```bash
# Verify cluster membership
curl http://localhost:8081/api/stats | jq '.stats'

# Check if node is in cluster
# If not, rejoin:
curl -X POST http://localhost:8081/api/join \
  -H "Content-Type: application/json" \
  -d '{"node_id":"node2","addr":"localhost:7001"}'
```

---

## Next Steps

1. **Read the full README.md** for detailed architecture and design
2. **Check DEPLOYMENT_GUIDE.md** for production deployment
3. **Review TROUBLESHOOTING.md** for common issues
4. **Try EXAMPLES.md** for more usage patterns

---

## Learning Resources

### Understanding Raft
- [Raft Consensus Algorithm](https://raft.github.io/)
- [Visual Explanation](http://thesecretlivesofdata.com/raft/)
- [Raft Paper](https://raft.github.io/raft.pdf)

### Go Distributed Systems
- [HashiCorp Raft](https://github.com/hashicorp/raft)
- [Distributed Systems in Go](https://golang.org/)

### Reference Implementations
- [etcd](https://github.com/etcd-io/etcd)
- [Consul](https://github.com/hashicorp/consul)
- [CockroachDB](https://github.com/cockroachdb/cockroach)

---

## Questions?

See the **FAQ section** in TROUBLESHOOTING.md or refer to the complete documentation in README.md.

Happy learning! 🚀

---

Last updated: 2025-10-22
Go version: 1.24+
