package main

import (
	"flag"
	"log"
	"os"
	"os/signal"
	"syscall"

	"github.com/yourusername/kvstore/internal/server"
)

func main() {
	nodeID := flag.String("node-id", getEnv("NODE_ID", "node1"), "Node ID")
	raftAddr := flag.String("raft-addr", getEnv("RAFT_ADDR", "localhost:7000"), "Raft bind address")
	httpAddr := flag.String("http-addr", getEnv("HTTP_ADDR", ":8080"), "HTTP API address")
	dataDir := flag.String("data-dir", getEnv("DATA_DIR", "./data"), "Data directory")
	bootstrap := flag.Bool("bootstrap", getEnv("BOOTSTRAP", "") != "", "Bootstrap cluster")
	join := flag.String("join", getEnv("JOIN", ""), "Join existing cluster")
	flag.Parse()

	log.Printf("=== Starting Distributed KV Store ===")
	log.Printf("Node ID: %s", *nodeID)
	log.Printf("Raft address: %s", *raftAddr)
	log.Printf("HTTP address: %s", *httpAddr)
	log.Printf("Data directory: %s", *dataDir)
	log.Printf("Bootstrap: %v", *bootstrap)
	if *join != "" {
		log.Printf("Join: %s", *join)
	}

	// Initialize node
	config := &server.Config{
		NodeID:    *nodeID,
		DataDir:   *dataDir,
		RaftAddr:  *raftAddr,
		Bootstrap: *bootstrap,
	}

	node, err := server.NewNode(config)
	if err != nil {
		log.Fatalf("Failed to create node: %v", err)
	}
	defer node.Close()

	log.Printf("Node %s started successfully", *nodeID)

	// If join address is provided, join the cluster
	if *join != "" {
		log.Printf("Attempting to join cluster at %s", *join)
		// In a real implementation, you would make an HTTP request to the leader
		// to join the cluster. For this educational example, nodes can be manually
		// joined using the /api/join endpoint
		log.Printf("To join the cluster, make a POST request to the leader:")
		log.Printf("curl -X POST http://LEADER_ADDR/api/join -H 'Content-Type: application/json' -d '{\"node_id\":\"%s\",\"addr\":\"%s\"}'", *nodeID, *raftAddr)
	}

	// Start HTTP server
	httpServer := server.NewHTTPServer(*httpAddr, node)
	go func() {
		if err := httpServer.Start(); err != nil {
			log.Printf("HTTP server error: %v", err)
		}
	}()

	log.Printf("HTTP API listening on %s", *httpAddr)
	log.Printf("Node ready to serve requests")

	// Wait for interrupt signal
	quit := make(chan os.Signal, 1)
	signal.Notify(quit, syscall.SIGINT, syscall.SIGTERM)
	<-quit

	log.Println("Shutting down gracefully...")
	httpServer.Close()
	log.Println("Shutdown complete")
}

func getEnv(key, fallback string) string {
	if value := os.Getenv(key); value != "" {
		return value
	}
	return fallback
}
