#!/bin/bash

set -e

echo "=== Distributed KV Store Demo ==="
echo ""

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

# Build the binary
echo -e "${YELLOW}Building kvstore...${NC}"
go build -o kvstore ./cmd/kvstore
echo -e "${GREEN}Build complete!${NC}"
echo ""

# Clean up old data
echo -e "${YELLOW}Cleaning up old data...${NC}"
rm -rf ./data
echo ""

# Function to wait for node to be ready
wait_for_node() {
    local port=$1
    local max_attempts=30
    local attempt=0

    while [ $attempt -lt $max_attempts ]; do
        if curl -s http://localhost:$port/health > /dev/null 2>&1; then
            return 0
        fi
        attempt=$((attempt + 1))
        sleep 1
    done
    return 1
}

# Start nodes in background
echo -e "${YELLOW}Starting 3-node cluster...${NC}"

echo "Starting node1 (leader)..."
./kvstore \
  --node-id=node1 \
  --raft-addr=localhost:7000 \
  --http-addr=:8081 \
  --data-dir=./data/node1 \
  --bootstrap > ./data/node1.log 2>&1 &
NODE1_PID=$!

echo "Starting node2..."
./kvstore \
  --node-id=node2 \
  --raft-addr=localhost:7001 \
  --http-addr=:8082 \
  --data-dir=./data/node2 > ./data/node2.log 2>&1 &
NODE2_PID=$!

echo "Starting node3..."
./kvstore \
  --node-id=node3 \
  --raft-addr=localhost:7002 \
  --http-addr=:8083 \
  --data-dir=./data/node3 > ./data/node3.log 2>&1 &
NODE3_PID=$!

# Wait for nodes to start
echo "Waiting for nodes to start..."
wait_for_node 8081 && echo "Node1 ready" || (echo "Node1 failed to start" && exit 1)
wait_for_node 8082 && echo "Node2 ready" || (echo "Node2 failed to start" && exit 1)
wait_for_node 8083 && echo "Node3 ready" || (echo "Node3 failed to start" && exit 1)

echo -e "${GREEN}All nodes started!${NC}"
echo ""

# Give leader a moment to be elected
sleep 3

# Join nodes to cluster
echo -e "${YELLOW}Joining nodes to cluster...${NC}"
curl -s -X POST http://localhost:8081/api/join \
  -H "Content-Type: application/json" \
  -d '{"node_id":"node2","addr":"localhost:7001"}' | jq '.'

curl -s -X POST http://localhost:8081/api/join \
  -H "Content-Type: application/json" \
  -d '{"node_id":"node3","addr":"localhost:7002"}' | jq '.'

echo -e "${GREEN}Cluster formed!${NC}"
echo ""

# Check cluster status
echo -e "${YELLOW}Checking cluster status...${NC}"
echo "Node 1 stats:"
curl -s http://localhost:8081/api/stats | jq '.is_leader, .leader'
echo ""

# Perform operations
echo -e "${YELLOW}Performing SET operations...${NC}"
curl -s -X POST http://localhost:8081/api/set \
  -H "Content-Type: application/json" \
  -d '{"key":"hello","value":"world"}' | jq '.'

curl -s -X POST http://localhost:8081/api/set \
  -H "Content-Type: application/json" \
  -d '{"key":"user:1","value":"alice"}' | jq '.'

curl -s -X POST http://localhost:8081/api/set \
  -H "Content-Type: application/json" \
  -d '{"key":"user:2","value":"bob"}' | jq '.'

curl -s -X POST http://localhost:8081/api/set \
  -H "Content-Type: application/json" \
  -d '{"key":"config:timeout","value":"30s"}' | jq '.'
echo ""

# Wait for replication
sleep 2

echo -e "${YELLOW}Reading data from different nodes...${NC}"
echo "Read 'hello' from node1:"
curl -s http://localhost:8081/api/get?key=hello | jq '.'

echo "Read 'hello' from node2:"
curl -s http://localhost:8082/api/get?key=hello | jq '.'

echo "Read 'hello' from node3:"
curl -s http://localhost:8083/api/get?key=hello | jq '.'
echo ""

echo -e "${YELLOW}Listing keys with prefix 'user:'...${NC}"
curl -s http://localhost:8081/api/list?prefix=user: | jq '.'
echo ""

echo -e "${YELLOW}Deleting a key...${NC}"
curl -s -X DELETE http://localhost:8081/api/delete?key=hello | jq '.'

echo "Verify deletion:"
curl -s http://localhost:8081/api/get?key=hello | jq '.'
echo ""

echo -e "${YELLOW}Testing fault tolerance - killing node3...${NC}"
kill $NODE3_PID
sleep 2

echo "Writing data with only 2 nodes (should still work):"
curl -s -X POST http://localhost:8081/api/set \
  -H "Content-Type: application/json" \
  -d '{"key":"resilient","value":"data"}' | jq '.'

echo "Reading from node1:"
curl -s http://localhost:8081/api/get?key=resilient | jq '.'
echo ""

echo -e "${GREEN}Demo complete!${NC}"
echo ""
echo -e "${YELLOW}Cluster is still running. Try these commands:${NC}"
echo ""
echo "# Set a value"
echo "curl -X POST http://localhost:8081/api/set -H 'Content-Type: application/json' -d '{\"key\":\"test\",\"value\":\"data\"}'"
echo ""
echo "# Get a value"
echo "curl http://localhost:8081/api/get?key=test"
echo ""
echo "# List all keys"
echo "curl http://localhost:8081/api/list?prefix="
echo ""
echo "# Check stats"
echo "curl http://localhost:8081/api/stats | jq '.'"
echo ""
echo -e "${RED}Press Ctrl+C to stop the cluster${NC}"

# Cleanup function
cleanup() {
    echo ""
    echo -e "${YELLOW}Shutting down cluster...${NC}"
    kill $NODE1_PID $NODE2_PID 2>/dev/null || true
    echo -e "${GREEN}Cleanup complete${NC}"
    exit 0
}

trap cleanup INT TERM

# Wait for interrupt
wait
