package raft

import (
	"encoding/json"
	"io"

	"github.com/hashicorp/raft"
	"github.com/yourusername/kvstore/internal/storage"
)

// Command represents a state machine command
type Command struct {
	Op    string `json:"op"`    // set, delete
	Key   string `json:"key"`
	Value string `json:"value"`
}

// FSM is the finite state machine for Raft
type FSM struct {
	storage storage.Storage
}

// NewFSM creates a new FSM
func NewFSM(store storage.Storage) *FSM {
	return &FSM{storage: store}
}

// Apply applies a Raft log entry to the FSM
func (f *FSM) Apply(log *raft.Log) interface{} {
	var cmd Command
	if err := json.Unmarshal(log.Data, &cmd); err != nil {
		return err
	}

	switch cmd.Op {
	case "set":
		return f.storage.Set(cmd.Key, cmd.Value)
	case "delete":
		return f.storage.Delete(cmd.Key)
	default:
		return nil
	}
}

// Snapshot returns a snapshot of the FSM
func (f *FSM) Snapshot() (raft.FSMSnapshot, error) {
	return &FSMSnapshot{storage: f.storage}, nil
}

// Restore restores the FSM from a snapshot
func (f *FSM) Restore(snapshot io.ReadCloser) error {
	defer snapshot.Close()
	return f.storage.Restore(snapshot)
}

// FSMSnapshot represents a snapshot of the FSM
type FSMSnapshot struct {
	storage storage.Storage
}

// Persist writes the snapshot to the sink
func (f *FSMSnapshot) Persist(sink raft.SnapshotSink) error {
	defer sink.Close()
	return f.storage.Snapshot(sink)
}

// Release is called when we are finished with the snapshot
func (f *FSMSnapshot) Release() {}
