package server

import (
	"encoding/json"
	"log"
	"net/http"

	"github.com/go-chi/chi/v5"
	"github.com/go-chi/chi/v5/middleware"
	"github.com/yourusername/kvstore/internal/storage"
)

// HTTPServer wraps the HTTP server for the node
type HTTPServer struct {
	node   *Node
	server *http.Server
}

// NewHTTPServer creates a new HTTP server
func NewHTTPServer(addr string, node *Node) *HTTPServer {
	r := chi.NewRouter()

	// Middleware
	r.Use(middleware.Logger)
	r.Use(middleware.Recoverer)
	r.Use(middleware.RequestID)

	// API routes
	r.Route("/api", func(r chi.Router) {
		r.Get("/get", handleGet(node))
		r.Post("/set", handleSet(node))
		r.Delete("/delete", handleDelete(node))
		r.Get("/list", handleList(node))
		r.Post("/join", handleJoin(node))
		r.Get("/stats", handleStats(node))
	})

	// Health check
	r.Get("/health", handleHealth(node))

	server := &http.Server{
		Addr:    addr,
		Handler: r,
	}

	return &HTTPServer{
		node:   node,
		server: server,
	}
}

// Start starts the HTTP server
func (s *HTTPServer) Start() error {
	log.Printf("Starting HTTP server on %s", s.server.Addr)
	return s.server.ListenAndServe()
}

// Close gracefully shuts down the HTTP server
func (s *HTTPServer) Close() error {
	return s.server.Close()
}

// Request and Response types

type SetRequest struct {
	Key   string `json:"key"`
	Value string `json:"value"`
}

type SetResponse struct {
	Success bool   `json:"success"`
	Error   string `json:"error,omitempty"`
}

type GetResponse struct {
	Value string `json:"value,omitempty"`
	Found bool   `json:"found"`
	Error string `json:"error,omitempty"`
}

type DeleteResponse struct {
	Success bool   `json:"success"`
	Error   string `json:"error,omitempty"`
}

type ListResponse struct {
	Pairs map[string]string `json:"pairs"`
	Error string            `json:"error,omitempty"`
}

type JoinRequest struct {
	NodeID string `json:"node_id"`
	Addr   string `json:"addr"`
}

type JoinResponse struct {
	Success bool   `json:"success"`
	Error   string `json:"error,omitempty"`
}

type StatsResponse struct {
	Stats    map[string]string `json:"stats"`
	IsLeader bool              `json:"is_leader"`
	Leader   string            `json:"leader"`
}

type HealthResponse struct {
	Status   string `json:"status"`
	NodeID   string `json:"node_id"`
	IsLeader bool   `json:"is_leader"`
	Leader   string `json:"leader"`
}

// Handlers

func handleGet(node *Node) http.HandlerFunc {
	return func(w http.ResponseWriter, r *http.Request) {
		key := r.URL.Query().Get("key")
		if key == "" {
			http.Error(w, "key parameter required", http.StatusBadRequest)
			return
		}

		value, err := node.Get(key)
		resp := GetResponse{
			Found: err == nil,
		}

		if err == nil {
			resp.Value = value
		} else if err != storage.ErrKeyNotFound {
			resp.Error = err.Error()
		}

		w.Header().Set("Content-Type", "application/json")
		json.NewEncoder(w).Encode(resp)
	}
}

func handleSet(node *Node) http.HandlerFunc {
	return func(w http.ResponseWriter, r *http.Request) {
		var req SetRequest
		if err := json.NewDecoder(r.Body).Decode(&req); err != nil {
			http.Error(w, err.Error(), http.StatusBadRequest)
			return
		}

		if req.Key == "" {
			http.Error(w, "key is required", http.StatusBadRequest)
			return
		}

		err := node.Set(req.Key, req.Value)
		resp := SetResponse{
			Success: err == nil,
		}

		if err != nil {
			resp.Error = err.Error()
			w.WriteHeader(http.StatusInternalServerError)
		}

		w.Header().Set("Content-Type", "application/json")
		json.NewEncoder(w).Encode(resp)
	}
}

func handleDelete(node *Node) http.HandlerFunc {
	return func(w http.ResponseWriter, r *http.Request) {
		key := r.URL.Query().Get("key")
		if key == "" {
			http.Error(w, "key parameter required", http.StatusBadRequest)
			return
		}

		err := node.Delete(key)
		resp := DeleteResponse{
			Success: err == nil,
		}

		if err != nil {
			resp.Error = err.Error()
			w.WriteHeader(http.StatusInternalServerError)
		}

		w.Header().Set("Content-Type", "application/json")
		json.NewEncoder(w).Encode(resp)
	}
}

func handleList(node *Node) http.HandlerFunc {
	return func(w http.ResponseWriter, r *http.Request) {
		prefix := r.URL.Query().Get("prefix")

		pairs, err := node.List(prefix)
		resp := ListResponse{
			Pairs: pairs,
		}

		if err != nil {
			resp.Error = err.Error()
			w.WriteHeader(http.StatusInternalServerError)
		}

		w.Header().Set("Content-Type", "application/json")
		json.NewEncoder(w).Encode(resp)
	}
}

func handleJoin(node *Node) http.HandlerFunc {
	return func(w http.ResponseWriter, r *http.Request) {
		var req JoinRequest
		if err := json.NewDecoder(r.Body).Decode(&req); err != nil {
			http.Error(w, err.Error(), http.StatusBadRequest)
			return
		}

		if req.NodeID == "" || req.Addr == "" {
			http.Error(w, "node_id and addr are required", http.StatusBadRequest)
			return
		}

		err := node.Join(req.NodeID, req.Addr)
		resp := JoinResponse{
			Success: err == nil,
		}

		if err != nil {
			resp.Error = err.Error()
			w.WriteHeader(http.StatusInternalServerError)
		}

		w.Header().Set("Content-Type", "application/json")
		json.NewEncoder(w).Encode(resp)
	}
}

func handleStats(node *Node) http.HandlerFunc {
	return func(w http.ResponseWriter, r *http.Request) {
		resp := StatsResponse{
			Stats:    node.Stats(),
			IsLeader: node.IsLeader(),
			Leader:   node.LeaderAddr(),
		}

		w.Header().Set("Content-Type", "application/json")
		json.NewEncoder(w).Encode(resp)
	}
}

func handleHealth(node *Node) http.HandlerFunc {
	return func(w http.ResponseWriter, r *http.Request) {
		resp := HealthResponse{
			Status:   "ok",
			NodeID:   node.config.NodeID,
			IsLeader: node.IsLeader(),
			Leader:   node.LeaderAddr(),
		}

		w.Header().Set("Content-Type", "application/json")
		json.NewEncoder(w).Encode(resp)
	}
}
