package server

import (
	"encoding/json"
	"fmt"
	"net"
	"os"
	"path/filepath"
	"time"

	"github.com/hashicorp/raft"
	raftboltdb "github.com/hashicorp/raft-boltdb"
	raftimpl "github.com/yourusername/kvstore/internal/raft"
	"github.com/yourusername/kvstore/internal/storage"
)

// Node represents a single node in the distributed KV store
type Node struct {
	raft    *raft.Raft
	fsm     *raftimpl.FSM
	storage storage.Storage
	config  *Config
}

// Config holds the node configuration
type Config struct {
	NodeID    string
	DataDir   string
	RaftAddr  string
	Bootstrap bool
}

// NewNode creates a new node instance
func NewNode(config *Config) (*Node, error) {
	// Create storage
	store := storage.NewMemoryStorage()

	// Create FSM
	fsm := raftimpl.NewFSM(store)

	// Setup Raft configuration
	raftConfig := raft.DefaultConfig()
	raftConfig.LocalID = raft.ServerID(config.NodeID)

	// Create data directory if it doesn't exist
	if err := os.MkdirAll(config.DataDir, 0755); err != nil {
		return nil, err
	}

	// Create Raft log store
	logStore, err := raftboltdb.NewBoltStore(filepath.Join(config.DataDir, "raft-log.db"))
	if err != nil {
		return nil, err
	}

	// Create stable store
	stableStore, err := raftboltdb.NewBoltStore(filepath.Join(config.DataDir, "raft-stable.db"))
	if err != nil {
		return nil, err
	}

	// Create snapshot store
	snapshotStore, err := raft.NewFileSnapshotStore(config.DataDir, 2, os.Stderr)
	if err != nil {
		return nil, err
	}

	// Setup Raft transport
	addr, err := net.ResolveTCPAddr("tcp", config.RaftAddr)
	if err != nil {
		return nil, err
	}

	transport, err := raft.NewTCPTransport(config.RaftAddr, addr, 3, 10*time.Second, os.Stderr)
	if err != nil {
		return nil, err
	}

	// Create Raft instance
	r, err := raft.NewRaft(raftConfig, fsm, logStore, stableStore, snapshotStore, transport)
	if err != nil {
		return nil, err
	}

	// Bootstrap cluster if this is the first node
	if config.Bootstrap {
		configuration := raft.Configuration{
			Servers: []raft.Server{
				{
					ID:      raftConfig.LocalID,
					Address: transport.LocalAddr(),
				},
			},
		}
		r.BootstrapCluster(configuration)
	}

	return &Node{
		raft:    r,
		fsm:     fsm,
		storage: store,
		config:  config,
	}, nil
}

// Get retrieves a value by key (can be served from any node)
func (n *Node) Get(key string) (string, error) {
	return n.storage.Get(key)
}

// Set stores a key-value pair (must go through Raft)
func (n *Node) Set(key, value string) error {
	if n.raft.State() != raft.Leader {
		return fmt.Errorf("not leader")
	}

	cmd := raftimpl.Command{
		Op:    "set",
		Key:   key,
		Value: value,
	}

	data, err := json.Marshal(cmd)
	if err != nil {
		return err
	}

	future := n.raft.Apply(data, 10*time.Second)
	return future.Error()
}

// Delete removes a key-value pair (must go through Raft)
func (n *Node) Delete(key string) error {
	if n.raft.State() != raft.Leader {
		return fmt.Errorf("not leader")
	}

	cmd := raftimpl.Command{
		Op:  "delete",
		Key: key,
	}

	data, err := json.Marshal(cmd)
	if err != nil {
		return err
	}

	future := n.raft.Apply(data, 10*time.Second)
	return future.Error()
}

// List returns all key-value pairs with the given prefix
func (n *Node) List(prefix string) (map[string]string, error) {
	return n.storage.List(prefix)
}

// Join adds a new node to the cluster (must be called on leader)
func (n *Node) Join(nodeID, addr string) error {
	if n.raft.State() != raft.Leader {
		return fmt.Errorf("not leader")
	}

	future := n.raft.AddVoter(raft.ServerID(nodeID), raft.ServerAddress(addr), 0, 0)
	return future.Error()
}

// IsLeader returns true if this node is the leader
func (n *Node) IsLeader() bool {
	return n.raft.State() == raft.Leader
}

// LeaderAddr returns the address of the current leader
func (n *Node) LeaderAddr() string {
	addr, _ := n.raft.LeaderWithID()
	return string(addr)
}

// Stats returns node statistics
func (n *Node) Stats() map[string]string {
	return n.raft.Stats()
}

// Close shuts down the node
func (n *Node) Close() error {
	if err := n.raft.Shutdown().Error(); err != nil {
		return err
	}
	return n.storage.Close()
}
