package storage

import (
	"encoding/json"
	"errors"
	"io"
	"strings"
	"sync"
)

var ErrKeyNotFound = errors.New("key not found")

// MemoryStorage is a simple in-memory key-value store
type MemoryStorage struct {
	mu   sync.RWMutex
	data map[string]string
}

// NewMemoryStorage creates a new in-memory storage instance
func NewMemoryStorage() *MemoryStorage {
	return &MemoryStorage{
		data: make(map[string]string),
	}
}

// Get retrieves a value by key
func (s *MemoryStorage) Get(key string) (string, error) {
	s.mu.RLock()
	defer s.mu.RUnlock()

	value, ok := s.data[key]
	if !ok {
		return "", ErrKeyNotFound
	}
	return value, nil
}

// Set stores a key-value pair
func (s *MemoryStorage) Set(key, value string) error {
	s.mu.Lock()
	defer s.mu.Unlock()

	s.data[key] = value
	return nil
}

// Delete removes a key-value pair
func (s *MemoryStorage) Delete(key string) error {
	s.mu.Lock()
	defer s.mu.Unlock()

	delete(s.data, key)
	return nil
}

// List returns all key-value pairs with the given prefix
func (s *MemoryStorage) List(prefix string) (map[string]string, error) {
	s.mu.RLock()
	defer s.mu.RUnlock()

	result := make(map[string]string)
	for key, value := range s.data {
		if strings.HasPrefix(key, prefix) {
			result[key] = value
		}
	}
	return result, nil
}

// Snapshot writes the entire data store to a writer
func (s *MemoryStorage) Snapshot(w io.Writer) error {
	s.mu.RLock()
	defer s.mu.RUnlock()

	encoder := json.NewEncoder(w)
	return encoder.Encode(s.data)
}

// Restore loads the data store from a reader
func (s *MemoryStorage) Restore(r io.Reader) error {
	s.mu.Lock()
	defer s.mu.Unlock()

	decoder := json.NewDecoder(r)
	newData := make(map[string]string)
	if err := decoder.Decode(&newData); err != nil {
		return err
	}

	s.data = newData
	return nil
}

// Close performs any necessary cleanup (no-op for memory storage)
func (s *MemoryStorage) Close() error {
	return nil
}
