# Contributing to Observability Platform

Thank you for your interest in contributing! This guide explains how to get started.

## Development Setup

### Prerequisites

- Go 1.23 or later
- Docker & Docker Compose (for full stack deployment)
- Git

### Quick Start

```bash
# Clone the repository
git clone https://github.com/yourusername/observability-platform.git
cd observability-platform

# Download dependencies
go mod download

# Build the project
make build

# Run tests
make test

# Start development server
make run
```

## Code Organization

```
.
├── cmd/
│   └── server/           # Application entry point
├── internal/
│   ├── api/              # HTTP handlers
│   ├── logs/             # Log aggregation
│   ├── metrics/          # Metrics collection
│   ├── models/           # Data types
│   └── tracing/          # Distributed tracing
├── examples/             # Sample scripts
└── README.md             # Documentation
```

## Making Changes

### Adding a New Endpoint

1. Add handler to `internal/api/handlers.go`
2. Register route in `cmd/server/main.go`
3. Add tests in `internal/api/handlers_test.go`
4. Update API documentation in README.md

### Adding a New Feature

1. Follow the existing package structure
2. Write tests alongside implementation
3. Document the feature in README.md
4. Update examples if applicable

## Testing

```bash
# Run all tests
make test

# Run specific package tests
go test -v ./internal/metrics/...

# Run with coverage
go test -cover ./...

# Run with race detector
go test -race ./...

# Benchmark
go test -bench=. ./internal/metrics/
```

## Code Quality

```bash
# Format code
go fmt ./...

# Run go vet
go vet ./...

# Run linter (requires golangci-lint)
make lint
```

## Submitting Changes

1. Create a feature branch: `git checkout -b feature/your-feature`
2. Make your changes
3. Add tests for new functionality
4. Update documentation
5. Run `make check` to verify everything passes
6. Submit a pull request

## Documentation

- Keep README.md up-to-date
- Add code comments for complex logic
- Document API changes with examples
- Include tests as documentation

## Performance

When optimizing code:
- Profile before and after changes
- Benchmark critical paths
- Document performance improvements
- Consider memory usage

## Questions?

Open an issue on GitHub or refer to the main README.md for detailed documentation.
