package main

import (
	"context"
	"log"
	"net/http"
	"os"
	"os/signal"
	"syscall"
	"time"

	"github.com/gorilla/mux"
	"github.com/yourusername/observability-platform/internal/api"
	"github.com/yourusername/observability-platform/internal/logs"
	"github.com/yourusername/observability-platform/internal/metrics"
	"github.com/yourusername/observability-platform/internal/tracing"
)

func main() {
	if err := run(); err != nil {
		log.Fatalf("Application error: %v", err)
	}
}

func run() error {
	// Initialize components
	metricsStorage := metrics.NewMemoryStorage()
	metricsCollector := metrics.NewCollector(metricsStorage)

	logStorage := logs.NewMemoryLogStorage()
	logAggregator := logs.NewAggregator(logStorage, 10000)

	traceCollector, err := tracing.NewCollector("observability-platform")
	if err != nil {
		return err
	}

	// Start background workers
	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	// Start metrics collection
	go metricsCollector.Start(ctx, 15*time.Second)

	// Start log aggregation
	go logAggregator.Start(ctx)

	// Create API
	apiHandler := api.NewAPI(metricsCollector, logAggregator, traceCollector)

	// Setup router
	router := mux.NewRouter()

	// Metrics endpoints
	router.HandleFunc("/api/metrics/query", apiHandler.MetricsQueryHandler).Methods("POST")

	// Logs endpoints
	router.HandleFunc("/api/logs/ingest", logAggregator.IngestHandler).Methods("POST")
	router.HandleFunc("/api/logs/query", apiHandler.LogsQueryHandler).Methods("POST")

	// Tracing endpoints
	router.HandleFunc("/api/traces/{id}", apiHandler.TraceHandler).Methods("GET")
	router.HandleFunc("/api/servicemap", apiHandler.ServiceMapHandler).Methods("GET")

	// Health check
	router.HandleFunc("/health", func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusOK)
		w.Write([]byte("OK"))
	}).Methods("GET")

	// Start server
	port := os.Getenv("PORT")
	if port == "" {
		port = "8080"
	}

	srv := &http.Server{
		Addr:         ":" + port,
		Handler:      router,
		ReadTimeout:  15 * time.Second,
		WriteTimeout: 15 * time.Second,
		IdleTimeout:  60 * time.Second,
	}

	go func() {
		log.Printf("Observability Platform listening on :%s", port)
		if err := srv.ListenAndServe(); err != nil && err != http.ErrServerClosed {
			log.Fatalf("Server error: %v", err)
		}
	}()

	// Wait for interrupt signal
	sigCh := make(chan os.Signal, 1)
	signal.Notify(sigCh, os.Interrupt, syscall.SIGTERM)
	<-sigCh

	// Graceful shutdown
	log.Println("Shutting down gracefully...")
	shutdownCtx, shutdownCancel := context.WithTimeout(context.Background(), 30*time.Second)
	defer shutdownCancel()

	traceCollector.Close(shutdownCtx)
	return srv.Shutdown(shutdownCtx)
}
