package api

import (
	"encoding/json"
	"net/http"
	"time"

	"github.com/gorilla/mux"
	"github.com/yourusername/observability-platform/internal/logs"
	"github.com/yourusername/observability-platform/internal/metrics"
	"github.com/yourusername/observability-platform/internal/models"
	"github.com/yourusername/observability-platform/internal/tracing"
)

// API handles HTTP requests
type API struct {
	metricsCollector *metrics.Collector
	logAggregator    *logs.Aggregator
	traceCollector   *tracing.Collector
}

// NewAPI creates a new API handler
func NewAPI(
	metricsCollector *metrics.Collector,
	logAggregator *logs.Aggregator,
	traceCollector *tracing.Collector,
) *API {
	return &API{
		metricsCollector: metricsCollector,
		logAggregator:    logAggregator,
		traceCollector:   traceCollector,
	}
}

// MetricsQueryHandler handles metrics queries
func (api *API) MetricsQueryHandler(w http.ResponseWriter, r *http.Request) {
	var query models.TimeSeriesQuery
	if err := json.NewDecoder(r.Body).Decode(&query); err != nil {
		http.Error(w, "Invalid query", http.StatusBadRequest)
		return
	}

	points, err := api.metricsCollector.Query(query)
	if err != nil {
		http.Error(w, err.Error(), http.StatusInternalServerError)
		return
	}

	w.Header().Set("Content-Type", "application/json")
	json.NewEncoder(w).Encode(points)
}

// LogsQueryHandler handles log queries
func (api *API) LogsQueryHandler(w http.ResponseWriter, r *http.Request) {
	var query models.LogQuery
	if err := json.NewDecoder(r.Body).Decode(&query); err != nil {
		http.Error(w, "Invalid query", http.StatusBadRequest)
		return
	}

	// Set default limit
	if query.Limit == 0 {
		query.Limit = 100
	}

	logs, err := api.logAggregator.Search(query)
	if err != nil {
		http.Error(w, err.Error(), http.StatusInternalServerError)
		return
	}

	w.Header().Set("Content-Type", "application/json")
	json.NewEncoder(w).Encode(logs)
}

// TraceHandler retrieves a specific trace
func (api *API) TraceHandler(w http.ResponseWriter, r *http.Request) {
	vars := mux.Vars(r)
	traceID := vars["id"]

	trace, err := api.traceCollector.GetTrace(traceID)
	if err != nil {
		http.Error(w, "Trace not found", http.StatusNotFound)
		return
	}

	w.Header().Set("Content-Type", "application/json")
	json.NewEncoder(w).Encode(trace)
}

// ServiceMapHandler returns service dependency map
func (api *API) ServiceMapHandler(w http.ResponseWriter, r *http.Request) {
	// Parse time range from query params
	startTime := time.Now().Add(-1 * time.Hour)
	endTime := time.Now()

	if start := r.URL.Query().Get("start"); start != "" {
		if t, err := time.Parse(time.RFC3339, start); err == nil {
			startTime = t
		}
	}

	if end := r.URL.Query().Get("end"); end != "" {
		if t, err := time.Parse(time.RFC3339, end); err == nil {
			endTime = t
		}
	}

	serviceMap, err := api.traceCollector.BuildServiceMap(models.TimeRange{
		Start: startTime,
		End:   endTime,
	})
	if err != nil {
		http.Error(w, err.Error(), http.StatusInternalServerError)
		return
	}

	w.Header().Set("Content-Type", "application/json")
	json.NewEncoder(w).Encode(serviceMap)
}
