package logs

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"time"

	"github.com/yourusername/observability-platform/internal/models"
)

// Aggregator collects and stores logs
type Aggregator struct {
	storage LogStorage
	buffer  chan models.LogEntry
}

// LogStorage interface for log storage
type LogStorage interface {
	StoreBatch(logs []models.LogEntry) error
	Search(query models.LogQuery) ([]models.LogEntry, error)
}

// NewAggregator creates a new log aggregator
func NewAggregator(storage LogStorage, bufferSize int) *Aggregator {
	return &Aggregator{
		storage: storage,
		buffer:  make(chan models.LogEntry, bufferSize),
	}
}

// IngestHandler handles log ingestion via HTTP
func (la *Aggregator) IngestHandler(w http.ResponseWriter, r *http.Request) {
	var entry models.LogEntry
	if err := json.NewDecoder(r.Body).Decode(&entry); err != nil {
		http.Error(w, "Invalid log entry", http.StatusBadRequest)
		return
	}

	// Set timestamp if not provided
	if entry.Timestamp.IsZero() {
		entry.Timestamp = time.Now()
	}

	// Add to buffer
	select {
	case la.buffer <- entry:
		w.WriteHeader(http.StatusAccepted)
	default:
		http.Error(w, "Buffer full", http.StatusServiceUnavailable)
	}
}

// Start begins processing buffered logs
func (la *Aggregator) Start(ctx context.Context) {
	ticker := time.NewTicker(5 * time.Second)
	defer ticker.Stop()

	batch := make([]models.LogEntry, 0, 1000)

	for {
		select {
		case <-ctx.Done():
			// Flush remaining logs
			if len(batch) > 0 {
				la.storage.StoreBatch(batch)
			}
			return

		case entry := <-la.buffer:
			batch = append(batch, entry)

			// Flush when batch is full
			if len(batch) >= 1000 {
				if err := la.storage.StoreBatch(batch); err != nil {
					fmt.Printf("Failed to store logs: %v\n", err)
				}
				batch = batch[:0]
			}

		case <-ticker.C:
			// Periodic flush
			if len(batch) > 0 {
				if err := la.storage.StoreBatch(batch); err != nil {
					fmt.Printf("Failed to store logs: %v\n", err)
				}
				batch = batch[:0]
			}
		}
	}
}

// Search queries logs
func (la *Aggregator) Search(query models.LogQuery) ([]models.LogEntry, error) {
	return la.storage.Search(query)
}
