package logs

import (
	"strings"
	"sync"

	"github.com/yourusername/observability-platform/internal/models"
)

// MemoryLogStorage implements in-memory log storage
// In production, use Elasticsearch or Loki
type MemoryLogStorage struct {
	logs []models.LogEntry
	mu   sync.RWMutex
}

// NewMemoryLogStorage creates a new in-memory log storage
func NewMemoryLogStorage() *MemoryLogStorage {
	return &MemoryLogStorage{
		logs: make([]models.LogEntry, 0),
	}
}

// StoreBatch stores a batch of logs
func (s *MemoryLogStorage) StoreBatch(logs []models.LogEntry) error {
	s.mu.Lock()
	defer s.mu.Unlock()

	s.logs = append(s.logs, logs...)

	// Keep only last 10,000 logs (memory limit)
	if len(s.logs) > 10000 {
		s.logs = s.logs[len(s.logs)-10000:]
	}

	return nil
}

// Search searches logs
func (s *MemoryLogStorage) Search(query models.LogQuery) ([]models.LogEntry, error) {
	s.mu.RLock()
	defer s.mu.RUnlock()

	results := make([]models.LogEntry, 0)

	for _, log := range s.logs {
		// Filter by time range
		if log.Timestamp.Before(query.StartTime) || log.Timestamp.After(query.EndTime) {
			continue
		}

		// Filter by service
		if query.Service != "" && log.Service != query.Service {
			continue
		}

		// Filter by level
		if query.Level != "" && log.Level != query.Level {
			continue
		}

		// Full-text search in message
		if query.Query != "" && !strings.Contains(strings.ToLower(log.Message), strings.ToLower(query.Query)) {
			continue
		}

		results = append(results, log)

		// Limit results
		if len(results) >= query.Limit {
			break
		}
	}

	return results, nil
}
