package metrics

import (
	"context"
	"fmt"
	"io"
	"net/http"
	"sync"
	"time"

	"github.com/prometheus/client_golang/prometheus"
	"github.com/prometheus/common/expfmt"
	io_prometheus_client "github.com/prometheus/client_model/go"
	"github.com/yourusername/observability-platform/internal/models"
)

// Collector collects metrics from various sources
type Collector struct {
	storage  Storage
	registry *prometheus.Registry
	targets  []string
	mu       sync.RWMutex
}

// Storage interface for metrics storage
type Storage interface {
	Store(metrics []models.Metric) error
	Query(query models.TimeSeriesQuery) ([]models.MetricPoint, error)
}

// NewCollector creates a new metrics collector
func NewCollector(storage Storage) *Collector {
	return &Collector{
		storage:  storage,
		registry: prometheus.NewRegistry(),
		targets:  make([]string, 0),
	}
}

// AddTarget adds a scrape target
func (c *Collector) AddTarget(target string) {
	c.mu.Lock()
	defer c.mu.Unlock()
	c.targets = append(c.targets, target)
}

// Start begins metric collection
func (c *Collector) Start(ctx context.Context, interval time.Duration) {
	ticker := time.NewTicker(interval)
	defer ticker.Stop()

	for {
		select {
		case <-ctx.Done():
			return
		case <-ticker.C:
			c.scrapeAll(ctx)
		}
	}
}

// scrapeAll scrapes all targets
func (c *Collector) scrapeAll(ctx context.Context) {
	c.mu.RLock()
	targets := make([]string, len(c.targets))
	copy(targets, c.targets)
	c.mu.RUnlock()

	for _, target := range targets {
		go func(t string) {
			if err := c.Scrape(ctx, t); err != nil {
				fmt.Printf("Failed to scrape %s: %v\n", t, err)
			}
		}(target)
	}
}

// Scrape collects metrics from a Prometheus endpoint
func (c *Collector) Scrape(ctx context.Context, target string) error {
	resp, err := http.Get(target + "/metrics")
	if err != nil {
		return fmt.Errorf("failed to fetch metrics: %w", err)
	}
	defer resp.Body.Close()

	return c.parseAndStore(resp.Body)
}

// parseAndStore parses Prometheus format and stores metrics
func (c *Collector) parseAndStore(reader io.Reader) error {
	var parser expfmt.TextParser
	metricFamilies, err := parser.TextToMetricFamilies(reader)
	if err != nil {
		return fmt.Errorf("failed to parse metrics: %w", err)
	}

	metrics := make([]models.Metric, 0)
	timestamp := time.Now()

	for metricName, metricFamily := range metricFamilies {
		for _, metric := range metricFamily.GetMetric() {
			labels := extractLabels(metric)
			value := extractValue(metric, metricFamily.GetType())

			metrics = append(metrics, models.Metric{
				Name:      metricName,
				Labels:    labels,
				Value:     value,
				Timestamp: timestamp,
			})
		}
	}

	return c.storage.Store(metrics)
}

// extractLabels extracts label pairs from a metric
func extractLabels(metric *io_prometheus_client.Metric) map[string]string {
	labels := make(map[string]string)
	for _, label := range metric.GetLabel() {
		labels[label.GetName()] = label.GetValue()
	}
	return labels
}

// extractValue extracts the numeric value from a metric
func extractValue(metric *io_prometheus_client.Metric, metricType io_prometheus_client.MetricType) float64 {
	switch metricType {
	case io_prometheus_client.MetricType_COUNTER:
		return metric.GetCounter().GetValue()
	case io_prometheus_client.MetricType_GAUGE:
		return metric.GetGauge().GetValue()
	case io_prometheus_client.MetricType_HISTOGRAM:
		return metric.GetHistogram().GetSampleSum()
	case io_prometheus_client.MetricType_SUMMARY:
		return metric.GetSummary().GetSampleSum()
	default:
		return 0
	}
}

// Query retrieves metrics
func (c *Collector) Query(query models.TimeSeriesQuery) ([]models.MetricPoint, error) {
	return c.storage.Query(query)
}

// IngestMetric ingests a single metric (for push-based metrics)
func (c *Collector) IngestMetric(metric models.Metric) error {
	return c.storage.Store([]models.Metric{metric})
}
