package metrics

import (
	"fmt"
	"sync"
	"time"

	"github.com/yourusername/observability-platform/internal/models"
)

// MemoryStorage implements in-memory metrics storage
// In production, use ClickHouse or Prometheus
type MemoryStorage struct {
	metrics map[string][]models.Metric
	mu      sync.RWMutex
}

// NewMemoryStorage creates a new in-memory storage
func NewMemoryStorage() *MemoryStorage {
	return &MemoryStorage{
		metrics: make(map[string][]models.Metric),
	}
}

// Store stores metrics in memory
func (s *MemoryStorage) Store(metrics []models.Metric) error {
	s.mu.Lock()
	defer s.mu.Unlock()

	for _, metric := range metrics {
		key := buildKey(metric.Name, metric.Labels)
		s.metrics[key] = append(s.metrics[key], metric)

		// Keep only last 1 hour of data (memory limit)
		cutoff := time.Now().Add(-1 * time.Hour)
		s.metrics[key] = filterOldMetrics(s.metrics[key], cutoff)
	}

	return nil
}

// Query retrieves metrics matching the query
func (s *MemoryStorage) Query(query models.TimeSeriesQuery) ([]models.MetricPoint, error) {
	s.mu.RLock()
	defer s.mu.RUnlock()

	key := buildKey(query.Metric, query.Labels)
	metrics, exists := s.metrics[key]
	if !exists {
		return []models.MetricPoint{}, nil
	}

	// Filter by time range
	points := make([]models.MetricPoint, 0)
	for _, metric := range metrics {
		if metric.Timestamp.After(query.StartTime) && metric.Timestamp.Before(query.EndTime) {
			points = append(points, models.MetricPoint{
				Timestamp: metric.Timestamp,
				Value:     metric.Value,
			})
		}
	}

	// Downsample if step is specified
	if query.Step > 0 {
		points = downsample(points, query.Step)
	}

	return points, nil
}

// buildKey creates a unique key for a metric
func buildKey(name string, labels map[string]string) string {
	key := name
	for k, v := range labels {
		key += fmt.Sprintf(",%s=%s", k, v)
	}
	return key
}

// filterOldMetrics removes metrics older than cutoff
func filterOldMetrics(metrics []models.Metric, cutoff time.Time) []models.Metric {
	filtered := make([]models.Metric, 0)
	for _, metric := range metrics {
		if metric.Timestamp.After(cutoff) {
			filtered = append(filtered, metric)
		}
	}
	return filtered
}

// downsample reduces data points to match step interval
func downsample(points []models.MetricPoint, step time.Duration) []models.MetricPoint {
	if len(points) == 0 {
		return points
	}

	downsampled := make([]models.MetricPoint, 0)
	var bucket []models.MetricPoint
	bucketStart := points[0].Timestamp.Truncate(step)

	for _, point := range points {
		pointBucket := point.Timestamp.Truncate(step)

		if pointBucket.After(bucketStart) {
			// New bucket, aggregate previous
			if len(bucket) > 0 {
				downsampled = append(downsampled, aggregateBucket(bucket, bucketStart))
			}
			bucket = []models.MetricPoint{point}
			bucketStart = pointBucket
		} else {
			bucket = append(bucket, point)
		}
	}

	// Aggregate last bucket
	if len(bucket) > 0 {
		downsampled = append(downsampled, aggregateBucket(bucket, bucketStart))
	}

	return downsampled
}

// aggregateBucket computes average for a bucket
func aggregateBucket(points []models.MetricPoint, timestamp time.Time) models.MetricPoint {
	sum := 0.0
	for _, p := range points {
		sum += p.Value
	}
	return models.MetricPoint{
		Timestamp: timestamp,
		Value:     sum / float64(len(points)),
	}
}
