package models

import (
	"time"
)

// Metric represents a single metric data point
type Metric struct {
	Name      string            `json:"name"`
	Labels    map[string]string `json:"labels"`
	Value     float64           `json:"value"`
	Timestamp time.Time         `json:"timestamp"`
}

// MetricPoint represents a time-series point
type MetricPoint struct {
	Timestamp time.Time `json:"timestamp"`
	Value     float64   `json:"value"`
}

// TimeSeriesQuery represents a query for time-series data
type TimeSeriesQuery struct {
	Metric    string            `json:"metric"`
	Labels    map[string]string `json:"labels"`
	StartTime time.Time         `json:"start_time"`
	EndTime   time.Time         `json:"end_time"`
	Step      time.Duration     `json:"step"`
}

// LogEntry represents a log entry
type LogEntry struct {
	Timestamp time.Time              `json:"timestamp"`
	Level     string                 `json:"level"`
	Service   string                 `json:"service"`
	Message   string                 `json:"message"`
	Fields    map[string]interface{} `json:"fields"`
	TraceID   string                 `json:"trace_id,omitempty"`
}

// LogQuery represents a query for logs
type LogQuery struct {
	Service   string    `json:"service,omitempty"`
	Level     string    `json:"level,omitempty"`
	StartTime time.Time `json:"start_time"`
	EndTime   time.Time `json:"end_time"`
	Query     string    `json:"query,omitempty"` // Full-text search
	Limit     int       `json:"limit"`
}

// Span represents a distributed trace span
type Span struct {
	TraceID   string            `json:"trace_id"`
	SpanID    string            `json:"span_id"`
	ParentID  string            `json:"parent_id"`
	Service   string            `json:"service"`
	Operation string            `json:"operation"`
	StartTime time.Time         `json:"start_time"`
	Duration  time.Duration     `json:"duration"`
	Tags      map[string]string `json:"tags"`
	Logs      []SpanLog         `json:"logs"`
}

// SpanLog represents a log within a span
type SpanLog struct {
	Timestamp time.Time              `json:"timestamp"`
	Fields    map[string]interface{} `json:"fields"`
}

// Trace represents a complete distributed trace
type Trace struct {
	TraceID   string        `json:"trace_id"`
	Spans     []Span        `json:"spans"`
	Services  []string      `json:"services"`
	Duration  time.Duration `json:"duration"`
	StartTime time.Time     `json:"start_time"`
}

// TimeRange represents a time window
type TimeRange struct {
	Start time.Time `json:"start"`
	End   time.Time `json:"end"`
}

// ServiceMap represents service dependencies
type ServiceMap struct {
	Nodes        []ServiceNode       `json:"nodes"`
	Dependencies []ServiceDependency `json:"dependencies"`
}

// ServiceNode represents a service in the dependency graph
type ServiceNode struct {
	Name         string  `json:"name"`
	RequestCount int64   `json:"request_count"`
	ErrorRate    float64 `json:"error_rate"`
	AvgLatency   float64 `json:"avg_latency_ms"`
}

// ServiceDependency represents a dependency between services
type ServiceDependency struct {
	From         string  `json:"from"`
	To           string  `json:"to"`
	RequestCount int64   `json:"request_count"`
	ErrorRate    float64 `json:"error_rate"`
}

// AlertRule represents an alerting rule
type AlertRule struct {
	ID          string            `json:"id"`
	Name        string            `json:"name"`
	Query       string            `json:"query"`
	Threshold   float64           `json:"threshold"`
	Comparison  string            `json:"comparison"` // gt, lt, eq
	Duration    time.Duration     `json:"duration"`
	Severity    string            `json:"severity"`
	Annotations map[string]string `json:"annotations"`
}

// Alert represents a fired alert
type Alert struct {
	Rule     AlertRule `json:"rule"`
	Value    float64   `json:"value"`
	FiredAt  time.Time `json:"fired_at"`
	Resolved bool      `json:"resolved"`
}
