package tracing

import (
	"context"
	"fmt"
	"sync"
	"time"

	"github.com/yourusername/observability-platform/internal/models"
	"go.opentelemetry.io/otel"
	"go.opentelemetry.io/otel/exporters/jaeger"
	"go.opentelemetry.io/otel/sdk/resource"
	"go.opentelemetry.io/otel/sdk/trace"
	semconv "go.opentelemetry.io/otel/semconv/v1.4.0"
)

// Collector collects and stores traces
type Collector struct {
	spans    map[string][]models.Span
	mu       sync.RWMutex
	provider *trace.TracerProvider
}

// NewCollector creates a new trace collector
func NewCollector(serviceName string) (*Collector, error) {
	// Create Jaeger exporter
	exporter, err := jaeger.New(jaeger.WithCollectorEndpoint())
	if err != nil {
		return nil, err
	}

	// Create trace provider
	provider := trace.NewTracerProvider(
		trace.WithBatcher(exporter),
		trace.WithResource(resource.NewWithAttributes(
			semconv.SchemaURL,
			semconv.ServiceNameKey.String(serviceName),
		)),
	)

	otel.SetTracerProvider(provider)

	return &Collector{
		spans:    make(map[string][]models.Span),
		provider: provider,
	}, nil
}

// IngestSpan ingests a span
func (c *Collector) IngestSpan(span models.Span) error {
	c.mu.Lock()
	defer c.mu.Unlock()

	c.spans[span.TraceID] = append(c.spans[span.TraceID], span)
	return nil
}

// GetTrace retrieves a complete trace
func (c *Collector) GetTrace(traceID string) (*models.Trace, error) {
	c.mu.RLock()
	defer c.mu.RUnlock()

	spans, exists := c.spans[traceID]
	if !exists || len(spans) == 0 {
		return nil, fmt.Errorf("trace not found")
	}

	// Find root span (no parent)
	var startTime time.Time
	var duration time.Duration
	services := make(map[string]bool)

	for _, span := range spans {
		if span.ParentID == "" {
			startTime = span.StartTime
			duration = span.Duration
		}
		services[span.Service] = true
	}

	serviceList := make([]string, 0, len(services))
	for service := range services {
		serviceList = append(serviceList, service)
	}

	return &models.Trace{
		TraceID:   traceID,
		Spans:     spans,
		Services:  serviceList,
		Duration:  duration,
		StartTime: startTime,
	}, nil
}

// BuildServiceMap creates a dependency graph
func (c *Collector) BuildServiceMap(timeRange models.TimeRange) (*models.ServiceMap, error) {
	c.mu.RLock()
	defer c.mu.RUnlock()

	nodeStats := make(map[string]*models.ServiceNode)
	depStats := make(map[string]*models.ServiceDependency)

	// Analyze all traces
	for _, spans := range c.spans {
		for _, span := range spans {
			// Filter by time range
			if span.StartTime.Before(timeRange.Start) || span.StartTime.After(timeRange.End) {
				continue
			}

			// Update node stats
			if _, exists := nodeStats[span.Service]; !exists {
				nodeStats[span.Service] = &models.ServiceNode{
					Name: span.Service,
				}
			}
			node := nodeStats[span.Service]
			node.RequestCount++
			node.AvgLatency = (node.AvgLatency*float64(node.RequestCount-1) + float64(span.Duration.Milliseconds())) / float64(node.RequestCount)

			// Find parent span for dependency
			if span.ParentID != "" {
				parentSpan := c.findSpan(spans, span.ParentID)
				if parentSpan != nil && parentSpan.Service != span.Service {
					depKey := parentSpan.Service + "->" + span.Service
					if _, exists := depStats[depKey]; !exists {
						depStats[depKey] = &models.ServiceDependency{
							From: parentSpan.Service,
							To:   span.Service,
						}
					}
					depStats[depKey].RequestCount++
				}
			}
		}
	}

	// Convert maps to slices
	nodes := make([]models.ServiceNode, 0, len(nodeStats))
	for _, node := range nodeStats {
		nodes = append(nodes, *node)
	}

	deps := make([]models.ServiceDependency, 0, len(depStats))
	for _, dep := range depStats {
		deps = append(deps, *dep)
	}

	return &models.ServiceMap{
		Nodes:        nodes,
		Dependencies: deps,
	}, nil
}

// findSpan finds a span by ID
func (c *Collector) findSpan(spans []models.Span, spanID string) *models.Span {
	for _, span := range spans {
		if span.SpanID == spanID {
			return &span
		}
	}
	return nil
}

// Close shuts down the trace provider
func (c *Collector) Close(ctx context.Context) error {
	return c.provider.Shutdown(ctx)
}
