# Container Orchestrator - API Examples

Complete examples for all REST API endpoints.

## Base URL

```
http://localhost:8080
```

## Authentication

Currently no authentication required. Add API keys for production.

---

## 1. Deploy Service

Create and deploy a new service with specified number of replicas.

### Basic Deployment (Minimal)

**Request:**
```bash
curl -X POST http://localhost:8080/api/deploy \
  -H "Content-Type: application/json" \
  -d '{
    "name": "nginx-web",
    "image": "nginx:alpine",
    "replicas": 2
  }'
```

**Response:**
```json
{
  "name": "nginx-web",
  "image": "nginx:alpine",
  "replicas": 2,
  "containers": [
    {
      "id": "a1b2c3d4e5f6",
      "name": "nginx-web-0",
      "service_name": "nginx-web",
      "node_id": "local-node",
      "image": "nginx:alpine",
      "status": "running",
      "health": "unknown",
      "created_at": "2024-01-15T10:00:00Z",
      "started_at": "2024-01-15T10:00:01Z"
    },
    {
      "id": "f6e5d4c3b2a1",
      "name": "nginx-web-1",
      "service_name": "nginx-web",
      "node_id": "local-node",
      "image": "nginx:alpine",
      "status": "running",
      "health": "unknown",
      "created_at": "2024-01-15T10:00:00Z",
      "started_at": "2024-01-15T10:00:01Z"
    }
  ],
  "health_check": {
    "type": "",
    "endpoint": "",
    "interval": 0,
    "timeout": 0,
    "retries": 0
  },
  "created_at": "2024-01-15T10:00:00Z"
}
```

### Full Featured Deployment

**Request:**
```bash
curl -X POST http://localhost:8080/api/deploy \
  -H "Content-Type: application/json" \
  -d '{
    "name": "api-server",
    "image": "node:18-alpine",
    "replicas": 3,
    "command": ["node", "server.js"],
    "env": {
      "NODE_ENV": "production",
      "DATABASE_URL": "postgresql://user:pass@db:5432/app",
      "LOG_LEVEL": "info",
      "PORT": "3000"
    },
    "ports": [
      {
        "container_port": 3000,
        "host_port": 3000,
        "protocol": "tcp"
      }
    ],
    "resources": {
      "cpu_shares": 1024,
      "memory_mb": 512
    },
    "health_check": {
      "type": "http",
      "endpoint": "http://localhost:3000/health",
      "interval": "10s",
      "timeout": "5s",
      "retries": 3
    }
  }'
```

**Field Explanations:**
- `name`: Unique service identifier (alphanumeric + hyphens)
- `image`: Docker image name (must exist in Docker Hub or local)
- `replicas`: Number of container instances (default: 1)
- `command`: Override container entrypoint
- `env`: Environment variables passed to containers
- `ports`: Port mappings (container → host)
- `resources`: CPU/memory limits
- `health_check`: Health monitoring configuration

### Deployment with TCP Health Check

**Request:**
```bash
curl -X POST http://localhost:8080/api/deploy \
  -H "Content-Type: application/json" \
  -d '{
    "name": "postgres-db",
    "image": "postgres:15-alpine",
    "replicas": 1,
    "env": {
      "POSTGRES_PASSWORD": "secretpassword",
      "POSTGRES_DB": "appdb"
    },
    "ports": [
      {
        "container_port": 5432,
        "host_port": 5432,
        "protocol": "tcp"
      }
    ],
    "resources": {
      "memory_mb": 1024
    },
    "health_check": {
      "type": "tcp",
      "endpoint": "localhost:5432",
      "interval": "10s",
      "timeout": "5s",
      "retries": 3
    }
  }'
```

---

## 2. List Services

Retrieve all deployed services with their current status.

**Request:**
```bash
curl http://localhost:8080/api/services
```

**Response:**
```json
[
  {
    "name": "nginx-web",
    "image": "nginx:alpine",
    "replicas": 2,
    "containers": [
      {
        "id": "a1b2c3d4e5f6",
        "name": "nginx-web-0",
        "service_name": "nginx-web",
        "node_id": "local-node",
        "image": "nginx:alpine",
        "status": "running",
        "health": "unknown",
        "created_at": "2024-01-15T10:00:00Z",
        "started_at": "2024-01-15T10:00:01Z"
      },
      {
        "id": "f6e5d4c3b2a1",
        "name": "nginx-web-1",
        "service_name": "nginx-web",
        "node_id": "local-node",
        "image": "nginx:alpine",
        "status": "running",
        "health": "unknown",
        "created_at": "2024-01-15T10:00:00Z",
        "started_at": "2024-01-15T10:00:01Z"
      }
    ],
    "health_check": {},
    "created_at": "2024-01-15T10:00:00Z"
  },
  {
    "name": "api-server",
    "image": "node:18-alpine",
    "replicas": 3,
    "containers": [
      {
        "id": "c1d2e3f4g5h6",
        "name": "api-server-0",
        "service_name": "api-server",
        "node_id": "local-node",
        "image": "node:18-alpine",
        "status": "running",
        "health": "healthy",
        "created_at": "2024-01-15T10:05:00Z",
        "started_at": "2024-01-15T10:05:01Z"
      }
    ],
    "health_check": {
      "type": "http",
      "endpoint": "http://localhost:3000/health",
      "interval": 10000000000,
      "timeout": 5000000000,
      "retries": 3
    },
    "created_at": "2024-01-15T10:05:00Z"
  }
]
```

**Using jq for filtering:**
```bash
# Get all service names
curl http://localhost:8080/api/services | jq '.[].name'

# Get container count per service
curl http://localhost:8080/api/services | jq '.[] | {name, container_count: (.containers | length)}'

# Get unhealthy containers
curl http://localhost:8080/api/services | jq '.[] | select(.containers[] | select(.health == "unhealthy"))'
```

---

## 3. Scale Service

Adjust the number of replicas for a service (scale up or down).

### Scale Up (Increase Replicas)

**Request:**
```bash
curl -X POST http://localhost:8080/api/services/api-server/scale \
  -H "Content-Type: application/json" \
  -d '{
    "replicas": 5
  }'
```

**Response:**
```json
{
  "message": "Scaling up..."
}
```

The orchestrator will:
1. Create 2 new containers (from 3 to 5)
2. Register them in the service
3. Start health monitoring
4. Log the scale event

### Scale Down (Decrease Replicas)

**Request:**
```bash
curl -X POST http://localhost:8080/api/services/api-server/scale \
  -H "Content-Type: application/json" \
  -d '{
    "replicas": 2
  }'
```

**Response:**
```json
{
  "message": "Scaling down..."
}
```

The orchestrator will:
1. Stop 3 containers (from 5 to 2)
2. Remove them from the service
3. Stop health monitoring
4. Log the scale event

### Scale to Zero (Shutdown Service)

**Request:**
```bash
curl -X POST http://localhost:8080/api/services/api-server/scale \
  -H "Content-Type: application/json" \
  -d '{
    "replicas": 0
  }'
```

---

## 4. Get Service Logs

Stream logs from a service's first container.

**Request:**
```bash
curl http://localhost:8080/api/services/api-server/logs
```

**Response:**
```
2024-01-15T10:05:00Z [INFO] Starting Node.js server
2024-01-15T10:05:01Z [INFO] Connected to database
2024-01-15T10:05:02Z [INFO] Server listening on port 3000
2024-01-15T10:05:10Z [DEBUG] GET /health
2024-01-15T10:05:20Z [DEBUG] GET /health
```

**Follow logs (tail):**
```bash
# With curl, we get the full log at once
# For real-time logs, run multiple times:
watch -n 1 'curl http://localhost:8080/api/services/api-server/logs | tail -20'
```

**Note:** Currently returns logs from first container only. For production, implement log aggregation across all replicas.

---

## 5. View Events

Get the event history (last 100 events) for auditing and debugging.

**Request:**
```bash
curl http://localhost:8080/api/events
```

**Response:**
```json
[
  {
    "timestamp": "2024-01-15T10:00:00Z",
    "type": "deployment",
    "service": "nginx-web",
    "container": "",
    "message": "Deployed 2 replicas"
  },
  {
    "timestamp": "2024-01-15T10:05:00Z",
    "type": "deployment",
    "service": "api-server",
    "container": "",
    "message": "Deployed 3 replicas"
  },
  {
    "timestamp": "2024-01-15T10:10:15Z",
    "type": "health_check",
    "service": "api-server",
    "container": "c1d2e3f4g5h6",
    "message": "Container failed health check, restarting..."
  },
  {
    "timestamp": "2024-01-15T10:15:00Z",
    "type": "scale",
    "service": "api-server",
    "container": "",
    "message": "Scaled from 3 to 5 replicas"
  }
]
```

**Useful queries with jq:**

```bash
# Get all deployments
curl http://localhost:8080/api/events | jq '.[] | select(.type == "deployment")'

# Get health check events for a service
curl http://localhost:8080/api/events | jq '.[] | select(.type == "health_check" and .service == "api-server")'

# Get last 10 events
curl http://localhost:8080/api/events | jq '.[-10:]'

# Get events for specific time period
curl http://localhost:8080/api/events | jq '.[] | select(.timestamp > "2024-01-15T10:10:00Z")'
```

---

## Error Responses

### 400 Bad Request - Missing Required Fields

**Request:**
```bash
curl -X POST http://localhost:8080/api/deploy \
  -H "Content-Type: application/json" \
  -d '{
    "image": "nginx:alpine"
  }'
```

**Response:**
```
HTTP/1.1 400 Bad Request
Content-Type: text/plain

Name and image are required
```

### 400 Bad Request - Invalid JSON

**Request:**
```bash
curl -X POST http://localhost:8080/api/deploy \
  -H "Content-Type: application/json" \
  -d 'invalid json'
```

**Response:**
```
HTTP/1.1 400 Bad Request
Content-Type: text/plain

Invalid request
```

### 404 Not Found - Service Not Found

**Request:**
```bash
curl http://localhost:8080/api/services/nonexistent/logs
```

**Response:**
```
HTTP/1.1 404 Not Found
Content-Type: text/plain

Service not found
```

### 500 Internal Server Error - Scheduling Failed

**Request:**
```bash
curl -X POST http://localhost:8080/api/deploy \
  -H "Content-Type: application/json" \
  -d '{
    "name": "huge-app",
    "image": "nginx:alpine",
    "replicas": 1,
    "resources": {
      "memory_mb": 999999
    }
  }'
```

**Response:**
```
HTTP/1.1 500 Internal Server Error
Content-Type: text/plain

no suitable node found
```

---

## Complete Workflow Examples

### Example 1: Deploy and Monitor a Web Application

```bash
#!/bin/bash

# Deploy nginx web server
echo "1. Deploying web server..."
curl -X POST http://localhost:8080/api/deploy \
  -H "Content-Type: application/json" \
  -d '{
    "name": "web",
    "image": "nginx:alpine",
    "replicas": 2,
    "ports": [{
      "container_port": 80,
      "host_port": 8080,
      "protocol": "tcp"
    }]
  }' | jq .

sleep 2

# Check services
echo -e "\n2. Listing services..."
curl http://localhost:8080/api/services | jq '.[] | {name, replicas, status: (.containers[0].status)}'

# Scale up
echo -e "\n3. Scaling to 4 replicas..."
curl -X POST http://localhost:8080/api/services/web/scale \
  -H "Content-Type: application/json" \
  -d '{"replicas": 4}' | jq .

sleep 2

# View events
echo -e "\n4. Recent events..."
curl http://localhost:8080/api/events | jq '.[-5:]'
```

### Example 2: Deploy with Health Checks

```bash
#!/bin/bash

# Deploy application with health endpoint
curl -X POST http://localhost:8080/api/deploy \
  -H "Content-Type: application/json" \
  -d '{
    "name": "api",
    "image": "myapp:latest",
    "replicas": 3,
    "ports": [{
      "container_port": 3000,
      "host_port": 3000,
      "protocol": "tcp"
    }],
    "health_check": {
      "type": "http",
      "endpoint": "http://localhost:3000/health",
      "interval": "10s",
      "timeout": "5s",
      "retries": 3
    }
  }' | jq .

# Wait for health checks
sleep 15

# Check health status
curl http://localhost:8080/api/services/api | jq '.containers[] | {name, health}'
```

### Example 3: Database with Resource Limits

```bash
#!/bin/bash

# Deploy PostgreSQL with resource constraints
curl -X POST http://localhost:8080/api/deploy \
  -H "Content-Type: application/json" \
  -d '{
    "name": "database",
    "image": "postgres:15-alpine",
    "replicas": 1,
    "env": {
      "POSTGRES_PASSWORD": "secure_password",
      "POSTGRES_DB": "appdb",
      "POSTGRES_INITDB_ARGS": "-c max_connections=100"
    },
    "ports": [{
      "container_port": 5432,
      "host_port": 5432,
      "protocol": "tcp"
    }],
    "resources": {
      "cpu_shares": 2048,
      "memory_mb": 1024
    },
    "health_check": {
      "type": "tcp",
      "endpoint": "localhost:5432",
      "interval": "10s",
      "timeout": "5s",
      "retries": 3
    }
  }' | jq .
```

---

## Tips and Tricks

### Using Python Requests

```python
import requests
import json

BASE_URL = "http://localhost:8080"

# Deploy service
response = requests.post(
    f"{BASE_URL}/api/deploy",
    json={
        "name": "app",
        "image": "nginx:alpine",
        "replicas": 2
    }
)
print(response.json())

# List services
services = requests.get(f"{BASE_URL}/api/services").json()
for service in services:
    print(f"{service['name']}: {len(service['containers'])} containers")

# Scale service
requests.post(
    f"{BASE_URL}/api/services/app/scale",
    json={"replicas": 5}
)

# Get events
events = requests.get(f"{BASE_URL}/api/events").json()
for event in events[-5:]:
    print(f"{event['timestamp']} - {event['type']}: {event['message']}")
```

### Using httpie (Easier than curl)

```bash
# Install httpie
pip install httpie

# Deploy service
http POST localhost:8080/api/deploy name=app image=nginx:alpine replicas:=2

# List services
http localhost:8080/api/services

# Scale service
http POST localhost:8080/api/services/app/scale replicas:=5

# Get events
http localhost:8080/api/events | jq '.'
```

---

**For more information, see README.md and QUICKSTART.md**
