# Container Orchestrator - Deployment Guide

Production deployment guide for various environments.

## Local Development

### Prerequisites
- Go 1.23+
- Docker
- Docker socket access (`/var/run/docker.sock`)

### Setup

```bash
# Clone/extract project
cd container-orchestrator

# Download dependencies
go mod download

# Build binary
make build

# Run locally
./bin/orchestrator

# API available at http://localhost:8080
```

### With Hot Reload (Development)

```bash
# Install air
go install github.com/air-verse/air@latest

# Run with auto-reload
air

# Code changes trigger automatic rebuild
```

---

## Docker Deployment

### Single Container

**Build image:**
```bash
docker build -t container-orchestrator:latest .
```

**Run container:**
```bash
docker run -d \
  --name orchestrator \
  -p 8080:8080 \
  -v /var/run/docker.sock:/var/run/docker.sock \
  --restart unless-stopped \
  container-orchestrator:latest
```

**Verify:**
```bash
# Check running
docker ps | grep orchestrator

# View logs
docker logs orchestrator

# Test API
curl http://localhost:8080/api/services
```

**Stop and cleanup:**
```bash
docker stop orchestrator
docker rm orchestrator
```

### Docker Compose (Recommended)

**Start services:**
```bash
docker-compose up -d
```

**View logs:**
```bash
docker-compose logs -f orchestrator
```

**Stop services:**
```bash
docker-compose down
```

**Rebuild after code changes:**
```bash
docker-compose up -d --build
```

**Environment variables in docker-compose.yml:**
```yaml
environment:
  - PORT=8080
  - LOG_LEVEL=info
```

---

## Kubernetes Deployment

### Prerequisites
- kubectl configured
- Docker image pushed to registry

### 1. Build and Push Image

```bash
# Set your registry
REGISTRY=your-registry.azurecr.io
IMAGE=$REGISTRY/container-orchestrator:latest

# Build for multi-platform
docker buildx build --platform linux/amd64,linux/arm64 \
  -t $IMAGE \
  --push .

# Or build for single platform
docker build -t $IMAGE .
docker push $IMAGE
```

### 2. Create Kubernetes Manifests

**deployment.yaml:**
```yaml
apiVersion: apps/v1
kind: Deployment
metadata:
  name: orchestrator
  labels:
    app: orchestrator
spec:
  replicas: 1
  selector:
    matchLabels:
      app: orchestrator
  template:
    metadata:
      labels:
        app: orchestrator
    spec:
      serviceAccountName: orchestrator
      containers:
      - name: orchestrator
        image: your-registry/container-orchestrator:latest
        imagePullPolicy: IfNotPresent
        ports:
        - containerPort: 8080
          name: api
        env:
        - name: PORT
          value: "8080"
        resources:
          requests:
            memory: "256Mi"
            cpu: "250m"
          limits:
            memory: "512Mi"
            cpu: "500m"
        livenessProbe:
          httpGet:
            path: /api/services
            port: 8080
          initialDelaySeconds: 30
          periodSeconds: 10
        readinessProbe:
          httpGet:
            path: /api/services
            port: 8080
          initialDelaySeconds: 10
          periodSeconds: 5
        volumeMounts:
        - name: docker-socket
          mountPath: /var/run/docker.sock
      volumes:
      - name: docker-socket
        hostPath:
          path: /var/run/docker.sock
          type: Socket
```

**service.yaml:**
```yaml
apiVersion: v1
kind: Service
metadata:
  name: orchestrator
  labels:
    app: orchestrator
spec:
  type: LoadBalancer
  ports:
  - port: 80
    targetPort: 8080
    name: api
  selector:
    app: orchestrator
```

**rbac.yaml:**
```yaml
apiVersion: v1
kind: ServiceAccount
metadata:
  name: orchestrator

---
apiVersion: rbac.authorization.k8s.io/v1
kind: ClusterRole
metadata:
  name: orchestrator
rules:
- apiGroups: [""]
  resources: ["pods"]
  verbs: ["get", "list", "watch"]
- apiGroups: [""]
  resources: ["services"]
  verbs: ["get", "list"]

---
apiVersion: rbac.authorization.k8s.io/v1
kind: ClusterRoleBinding
metadata:
  name: orchestrator
roleRef:
  apiGroup: rbac.authorization.k8s.io
  kind: ClusterRole
  name: orchestrator
subjects:
- kind: ServiceAccount
  name: orchestrator
```

### 3. Deploy to Kubernetes

```bash
# Apply manifests
kubectl apply -f rbac.yaml
kubectl apply -f deployment.yaml
kubectl apply -f service.yaml

# Verify deployment
kubectl get deployments
kubectl get pods
kubectl get services

# View logs
kubectl logs -f deployment/orchestrator

# Port forward for testing
kubectl port-forward svc/orchestrator 8080:80

# Test API
curl http://localhost:8080/api/services

# Cleanup
kubectl delete -f deployment.yaml service.yaml rbac.yaml
```

---

## Cloud Deployments

### AWS ECS / Fargate

**1. Create ECR repository:**
```bash
aws ecr create-repository --repository-name container-orchestrator
```

**2. Build and push image:**
```bash
# Get login token
aws ecr get-login-password --region us-east-1 | \
  docker login --username AWS --password-stdin YOUR_ACCOUNT.dkr.ecr.us-east-1.amazonaws.com

# Build and tag
docker build -t YOUR_ACCOUNT.dkr.ecr.us-east-1.amazonaws.com/container-orchestrator:latest .

# Push
docker push YOUR_ACCOUNT.dkr.ecr.us-east-1.amazonaws.com/container-orchestrator:latest
```

**3. Create ECS task definition (task-definition.json):**
```json
{
  "family": "orchestrator",
  "networkMode": "awsvpc",
  "requiresCompatibilities": ["FARGATE"],
  "cpu": "256",
  "memory": "512",
  "containerDefinitions": [
    {
      "name": "orchestrator",
      "image": "YOUR_ACCOUNT.dkr.ecr.us-east-1.amazonaws.com/container-orchestrator:latest",
      "essential": true,
      "portMappings": [
        {
          "containerPort": 8080,
          "hostPort": 8080,
          "protocol": "tcp"
        }
      ],
      "environment": [
        {
          "name": "PORT",
          "value": "8080"
        }
      ],
      "logConfiguration": {
        "logDriver": "awslogs",
        "options": {
          "awslogs-group": "/ecs/orchestrator",
          "awslogs-region": "us-east-1",
          "awslogs-stream-prefix": "ecs"
        }
      },
      "mountPoints": [
        {
          "sourceVolume": "docker-socket",
          "containerPath": "/var/run/docker.sock"
        }
      ]
    }
  ],
  "volumes": [
    {
      "name": "docker-socket",
      "host": {
        "sourcePath": "/var/run/docker.sock"
      }
    }
  ]
}
```

**4. Register task and create service:**
```bash
# Register task definition
aws ecs register-task-definition --cli-input-json file://task-definition.json

# Create CloudWatch log group
aws logs create-log-group --log-group-name /ecs/orchestrator

# Create ECS service
aws ecs create-service \
  --cluster default \
  --service-name orchestrator \
  --task-definition orchestrator \
  --desired-count 1 \
  --launch-type FARGATE \
  --network-configuration "awsvpcConfiguration={subnets=[subnet-xxx],assignPublicIp=ENABLED}"
```

### Google Cloud Run

**Build and deploy:**
```bash
# Build image
gcloud builds submit --tag gcr.io/PROJECT_ID/container-orchestrator

# Deploy to Cloud Run
gcloud run deploy orchestrator \
  --image gcr.io/PROJECT_ID/container-orchestrator \
  --platform managed \
  --region us-central1 \
  --memory 512Mi \
  --cpu 1 \
  --allow-unauthenticated
```

### Azure Container Instances

```bash
# Create resource group
az group create --name orchestrator --location eastus

# Create container
az container create \
  --resource-group orchestrator \
  --name orchestrator \
  --image myregistry.azurecr.io/container-orchestrator:latest \
  --cpu 1 \
  --memory 0.5 \
  --ports 8080 \
  --ip-address public
```

---

## Systemd Service (Linux)

Deploy as systemd service for persistent availability.

### 1. Create Service File

**File: `/etc/systemd/system/orchestrator.service`**
```ini
[Unit]
Description=Container Orchestrator
After=docker.service
Requires=docker.service

[Service]
Type=simple
User=orchestrator
Group=docker
WorkingDirectory=/opt/orchestrator
ExecStart=/opt/orchestrator/bin/orchestrator
Restart=always
RestartSec=10
StandardOutput=journal
StandardError=journal

Environment="PORT=8080"

[Install]
WantedBy=multi-user.target
```

### 2. Setup User and Install Binary

```bash
# Create user
sudo useradd -r -s /bin/false -d /opt/orchestrator orchestrator

# Create directory
sudo mkdir -p /opt/orchestrator/bin

# Copy binary
sudo cp bin/orchestrator /opt/orchestrator/bin/
sudo chown -R orchestrator:docker /opt/orchestrator

# Add user to docker group
sudo usermod -aG docker orchestrator

# Copy config (optional)
sudo cp docker-compose.yml /opt/orchestrator/
```

### 3. Enable and Start Service

```bash
# Reload systemd
sudo systemctl daemon-reload

# Enable service
sudo systemctl enable orchestrator

# Start service
sudo systemctl start orchestrator

# Check status
sudo systemctl status orchestrator

# View logs
sudo journalctl -u orchestrator -f

# Stop service
sudo systemctl stop orchestrator
```

---

## SSL/TLS with Nginx Reverse Proxy

### Nginx Configuration

**File: `/etc/nginx/sites-available/orchestrator`**
```nginx
server {
    listen 80;
    server_name orchestrator.example.com;
    return 301 https://$server_name$request_uri;
}

server {
    listen 443 ssl http2;
    server_name orchestrator.example.com;

    ssl_certificate /etc/letsencrypt/live/orchestrator.example.com/fullchain.pem;
    ssl_certificate_key /etc/letsencrypt/live/orchestrator.example.com/privkey.pem;

    # Security headers
    add_header Strict-Transport-Security "max-age=31536000; includeSubDomains" always;
    add_header X-Frame-Options "DENY" always;
    add_header X-Content-Type-Options "nosniff" always;

    location / {
        proxy_pass http://localhost:8080;
        proxy_http_version 1.1;
        proxy_set_header Upgrade $http_upgrade;
        proxy_set_header Connection 'upgrade';
        proxy_set_header Host $host;
        proxy_cache_bypass $http_upgrade;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto $scheme;
    }
}
```

**Enable site:**
```bash
sudo ln -s /etc/nginx/sites-available/orchestrator /etc/nginx/sites-enabled/
sudo nginx -t
sudo systemctl restart nginx
```

---

## Monitoring and Health Checks

### Health Check Endpoint

The API provides a simple health endpoint:
```bash
curl http://localhost:8080/api/services
```

### Prometheus Metrics (Add to Production)

To add Prometheus metrics:

1. Import prometheus client library
2. Register collectors
3. Expose `/metrics` endpoint

Example addition to main.go:
```go
import "github.com/prometheus/client_golang/prometheus/promhttp"

router.Handle("/metrics", promhttp.Handler())
```

### Logging Configuration

Redirect logs to syslog:
```bash
# Docker
docker run ... --log-driver syslog ...

# Kubernetes - already logs to stdout
kubectl logs deployment/orchestrator
```

---

## Backup and Disaster Recovery

### State Backup

Currently, state is in-memory. For production, implement persistent storage:

```go
// Save state periodically
func saveState(registry *registry.Registry) error {
    services := registry.List()
    data := json.Marshal(services)
    return os.WriteFile("state.json", data, 0644)
}

// Restore on startup
func restoreState(registry *registry.Registry) error {
    data, err := os.ReadFile("state.json")
    if err != nil { return err }

    var services []*models.Service
    json.Unmarshal(data, &services)

    for _, svc := range services {
        registry.Register(svc)
    }
    return nil
}
```

### Database Backend

For persistent state:
- SQLite (single-node)
- PostgreSQL (distributed)
- etcd (Kubernetes-native)

---

## Security Best Practices

1. **API Authentication**: Add JWT or API keys
   ```go
   func authMiddleware(next http.Handler) http.Handler {
       return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
           token := r.Header.Get("Authorization")
           if !validateToken(token) {
               http.Error(w, "Unauthorized", 401)
               return
           }
           next.ServeHTTP(w, r)
       })
   }
   ```

2. **TLS/HTTPS**: Always use reverse proxy with SSL

3. **Rate Limiting**: Protect against DoS
   ```go
   import "github.com/gorilla/handlers"
   router.Use(handlers.RateLimitMiddleware(100))
   ```

4. **Audit Logging**: Log all operations to file/syslog

5. **RBAC**: Implement role-based access control

6. **Secrets**: Never hardcode credentials
   - Use environment variables
   - Use secret management (Docker secrets, Kubernetes secrets)

---

## Troubleshooting Deployment

### Port Already in Use
```bash
# Find and kill process
lsof -i :8080
kill -9 <PID>

# Or use different port
PORT=9090 ./bin/orchestrator
```

### Docker Socket Permission Denied
```bash
# Fix permissions
sudo usermod -aG docker $USER
newgrp docker

# Or run with sudo
sudo ./bin/orchestrator
```

### Cannot Connect to Docker
```bash
# Test Docker access
docker ps

# Check socket exists
ls -la /var/run/docker.sock

# Verify daemon running
systemctl status docker
```

### High Memory Usage
```bash
# Monitor
docker stats orchestrator

# Reduce event log size (current: 100)
# Implement periodic cleanup
# Use persistent database instead of memory
```

---

## Performance Tuning

### Resource Allocation

```bash
# Docker
docker run --memory 512m --cpus 0.5 container-orchestrator:latest

# Kubernetes
resources:
  requests:
    memory: "256Mi"
    cpu: "250m"
  limits:
    memory: "512Mi"
    cpu: "500m"
```

### Concurrency Tuning

```go
// Adjust in code
const (
    MaxHealthCheckGoroutines = 100
    EventLogSize = 1000  // increase from 100
    CacheRefreshInterval = 30 * time.Second
)
```

---

**For more information, see README.md and QUICKSTART.md**
