# Container Orchestrator - Quick Start Guide

Get the Container Orchestrator up and running in minutes!

## Prerequisites

- Go 1.23+ (check: `go version`)
- Docker (check: `docker --version`)
- Docker socket access

## 5-Minute Startup

### Option 1: Docker Compose (Easiest)

```bash
# Start the orchestrator
docker-compose up -d

# Check it's running
curl http://localhost:8080/api/services

# View logs
docker-compose logs -f orchestrator

# Stop when done
docker-compose down
```

### Option 2: Build and Run

```bash
# Download dependencies
go mod download

# Build binary
make build

# Run orchestrator
make run

# In another terminal, deploy a service
curl -X POST http://localhost:8080/api/deploy \
  -H "Content-Type: application/json" \
  -d '{
    "name": "nginx",
    "image": "nginx:alpine",
    "replicas": 1
  }'
```

## Your First Deployment

### Deploy Nginx

```bash
curl -X POST http://localhost:8080/api/deploy \
  -H "Content-Type: application/json" \
  -d '{
    "name": "web",
    "image": "nginx:alpine",
    "replicas": 2,
    "ports": [
      {
        "container_port": 80,
        "host_port": 8080,
        "protocol": "tcp"
      }
    ]
  }'
```

Expected response:
```json
{
  "name": "web",
  "image": "nginx:alpine",
  "replicas": 2,
  "containers": [
    {
      "id": "abc123...",
      "name": "web-0",
      "status": "running",
      "health": "unknown"
    },
    {
      "id": "def456...",
      "name": "web-1",
      "status": "running",
      "health": "unknown"
    }
  ]
}
```

### Access Your Service

```bash
# Verify nginx is running
curl http://localhost:8080

# You should see the nginx welcome page!
```

## Common Operations

### List All Services

```bash
curl http://localhost:8080/api/services
```

### Scale a Service

```bash
# Scale nginx to 5 replicas
curl -X POST http://localhost:8080/api/services/web/scale \
  -H "Content-Type: application/json" \
  -d '{"replicas": 5}'
```

### View Service Logs

```bash
curl http://localhost:8080/api/services/web/logs
```

### View Events

```bash
curl http://localhost:8080/api/events
```

### Deploy with Health Checks

```bash
curl -X POST http://localhost:8080/api/deploy \
  -H "Content-Type: application/json" \
  -d '{
    "name": "api",
    "image": "node:18-alpine",
    "replicas": 3,
    "command": ["node", "server.js"],
    "ports": [
      {
        "container_port": 3000,
        "host_port": 3000,
        "protocol": "tcp"
      }
    ],
    "resources": {
      "cpu_shares": 512,
      "memory_mb": 256
    },
    "health_check": {
      "type": "http",
      "endpoint": "http://localhost:3000/health",
      "interval": "10s",
      "timeout": "5s",
      "retries": 3
    }
  }'
```

### Deploy with Environment Variables

```bash
curl -X POST http://localhost:8080/api/deploy \
  -H "Content-Type: application/json" \
  -d '{
    "name": "app",
    "image": "myapp:latest",
    "replicas": 2,
    "env": {
      "DATABASE_URL": "postgres://db:5432/app",
      "ENVIRONMENT": "production",
      "LOG_LEVEL": "info"
    }
  }'
```

## Testing Locally

### Run Tests

```bash
# All tests
go test ./...

# Integration tests (requires Docker)
go test -v ./tests/

# With coverage
go test -cover ./...
```

### Manual Testing

1. **Deploy test service**:
   ```bash
   curl -X POST http://localhost:8080/api/deploy \
     -H "Content-Type: application/json" \
     -d '{
       "name": "test",
       "image": "busybox:latest",
       "replicas": 1,
       "command": ["sleep", "1000"]
     }'
   ```

2. **Verify deployment**:
   ```bash
   curl http://localhost:8080/api/services | jq '.[] | select(.name=="test")'
   ```

3. **Scale up**:
   ```bash
   curl -X POST http://localhost:8080/api/services/test/scale \
     -H "Content-Type: application/json" \
     -d '{"replicas": 3}'
   ```

4. **Check events**:
   ```bash
   curl http://localhost:8080/api/events | jq '.'
   ```

## Troubleshooting

### Docker Socket Error

```
Error: permission denied while trying to connect to Docker daemon
```

Solution:
```bash
# Add current user to docker group
sudo usermod -aG docker $USER

# Refresh group membership
newgrp docker

# Test Docker access
docker ps
```

### Port Already in Use

```
Error: listen tcp :8080: bind: address already in use
```

Solution:
```bash
# Use different port
PORT=9090 make run

# Or kill existing process
lsof -i :8080 | grep LISTEN | awk '{print $2}' | xargs kill -9
```

### Cannot Pull Images

```
Error: Failed to pull image
```

Solution:
```bash
# Login to Docker Hub
docker login

# Or pull image locally first
docker pull nginx:alpine

# Then try deployment
```

## Next Steps

1. **Read the full README.md** for comprehensive documentation
2. **Check ARCHITECTURE.md** for system design details
3. **Review API examples** in API_EXAMPLES.md
4. **Explore source code** in `internal/` directory
5. **Run integration tests** with `go test ./tests/`

## Useful Commands Cheat Sheet

```bash
# Build & Run
make build              # Build binary
make run                # Build and run
make clean              # Clean build artifacts

# Testing
make test              # Run unit tests
make test-integration  # Run integration tests

# Docker
make docker-build      # Build Docker image
make docker-run        # Run in Docker
docker-compose up      # Start with Docker Compose

# Deployment
curl http://localhost:8080/api/deploy          # Deploy service
curl http://localhost:8080/api/services        # List services
curl http://localhost:8080/api/services/{name}/scale    # Scale
curl http://localhost:8080/api/services/{name}/logs     # Get logs
curl http://localhost:8080/api/events          # View events
```

## Example Workflows

### Complete Workflow: Deploy Multi-Replica Service with Health Checks

```bash
#!/bin/bash

# 1. Deploy service with 2 replicas
echo "Deploying service..."
curl -X POST http://localhost:8080/api/deploy \
  -H "Content-Type: application/json" \
  -d '{
    "name": "production-app",
    "image": "node:18-alpine",
    "replicas": 2,
    "ports": [{"container_port": 3000, "host_port": 3000, "protocol": "tcp"}],
    "health_check": {
      "type": "http",
      "endpoint": "http://localhost:3000/health",
      "interval": "10s",
      "timeout": "5s",
      "retries": 3
    }
  }'

sleep 2

# 2. Verify deployment
echo "Checking services..."
curl http://localhost:8080/api/services | jq '.'

sleep 3

# 3. Scale to 5 replicas
echo "Scaling up..."
curl -X POST http://localhost:8080/api/services/production-app/scale \
  -H "Content-Type: application/json" \
  -d '{"replicas": 5}'

sleep 2

# 4. View events
echo "Recent events..."
curl http://localhost:8080/api/events | jq '.[-5:]'

# 5. Scale down to 2
echo "Scaling down..."
curl -X POST http://localhost:8080/api/services/production-app/scale \
  -H "Content-Type: application/json" \
  -d '{"replicas": 2}'

echo "Done!"
```

Save as `test-workflow.sh`, then run:
```bash
chmod +x test-workflow.sh
./test-workflow.sh
```

---

**Happy Orchestrating!**

For more information, see README.md and ARCHITECTURE.md
