package main

import (
	"context"
	"fmt"
	"log"
	"net/http"
	"os"
	"os/signal"
	"syscall"
	"time"

	"github.com/gorilla/mux"
	"github.com/yourusername/container-orchestrator/internal/api"
	"github.com/yourusername/container-orchestrator/internal/docker"
	"github.com/yourusername/container-orchestrator/internal/health"
	"github.com/yourusername/container-orchestrator/internal/models"
	"github.com/yourusername/container-orchestrator/internal/registry"
	"github.com/yourusername/container-orchestrator/internal/scheduler"
)

func main() {
	if err := run(); err != nil {
		log.Fatalf("Application error: %v", err)
	}
}

func run() error {
	// Initialize components
	dockerClient, err := docker.NewClient()
	if err != nil {
		return fmt.Errorf("failed to create docker client: %w", err)
	}
	defer dockerClient.Close()

	sched := scheduler.NewScheduler()
	reg := registry.NewRegistry()
	checker := health.NewChecker()

	// Register local node
	localNode := &models.Node{
		ID:        "local-node",
		Address:   "localhost",
		CPUCores:  4,
		MemoryMB:  8192,
		Available: true,
		LastSeen:  time.Now(),
	}
	sched.RegisterNode(localNode)

	// Create API
	apiHandler := api.NewAPI(dockerClient, sched, reg, checker)

	// Setup router
	router := mux.NewRouter()
	router.HandleFunc("/api/deploy", apiHandler.DeployHandler).Methods("POST")
	router.HandleFunc("/api/services", apiHandler.ListServicesHandler).Methods("GET")
	router.HandleFunc("/api/services/{name}/scale", apiHandler.ScaleHandler).Methods("POST")
	router.HandleFunc("/api/services/{name}/logs", apiHandler.LogsHandler).Methods("GET")
	router.HandleFunc("/api/events", apiHandler.EventsHandler).Methods("GET")

	// Start server
	port := os.Getenv("PORT")
	if port == "" {
		port = "8080"
	}

	srv := &http.Server{
		Addr:         ":" + port,
		Handler:      router,
		ReadTimeout:  15 * time.Second,
		WriteTimeout: 15 * time.Second,
		IdleTimeout:  60 * time.Second,
	}

	// Start server in goroutine
	go func() {
		log.Printf("Orchestrator API listening on :%s", port)
		if err := srv.ListenAndServe(); err != nil && err != http.ErrServerClosed {
			log.Fatalf("Server error: %v", err)
		}
	}()

	// Wait for interrupt signal
	sigCh := make(chan os.Signal, 1)
	signal.Notify(sigCh, os.Interrupt, syscall.SIGTERM)
	<-sigCh

	// Graceful shutdown
	log.Println("Shutting down gracefully...")
	ctx, cancel := context.WithTimeout(context.Background(), 30*time.Second)
	defer cancel()

	return srv.Shutdown(ctx)
}
