package api

import (
	"context"
	"encoding/json"
	"fmt"
	"io"
	"net/http"
	"time"

	"github.com/gorilla/mux"
	"github.com/yourusername/container-orchestrator/internal/docker"
	"github.com/yourusername/container-orchestrator/internal/health"
	"github.com/yourusername/container-orchestrator/internal/models"
	"github.com/yourusername/container-orchestrator/internal/registry"
	"github.com/yourusername/container-orchestrator/internal/scheduler"
)

// API handles HTTP requests
type API struct {
	dockerClient  *docker.Client
	scheduler     *scheduler.Scheduler
	registry      *registry.Registry
	healthChecker *health.Checker
	eventLog      []models.Event
}

// NewAPI creates a new API handler
func NewAPI(
	dockerClient *docker.Client,
	sched *scheduler.Scheduler,
	reg *registry.Registry,
	checker *health.Checker,
) *API {
	return &API{
		dockerClient:  dockerClient,
		scheduler:     sched,
		registry:      reg,
		healthChecker: checker,
		eventLog:      make([]models.Event, 0),
	}
}

// DeployHandler handles service deployment
func (api *API) DeployHandler(w http.ResponseWriter, r *http.Request) {
	var req models.DeploymentRequest
	if err := json.NewDecoder(r.Body).Decode(&req); err != nil {
		http.Error(w, "Invalid request body", http.StatusBadRequest)
		return
	}

	// Validate request
	if req.Name == "" || req.Image == "" {
		http.Error(w, "Name and image are required", http.StatusBadRequest)
		return
	}

	if req.Replicas <= 0 {
		req.Replicas = 1
	}

	// Create service
	service := &models.Service{
		Name:        req.Name,
		Image:       req.Image,
		Replicas:    req.Replicas,
		Containers:  make([]models.Container, 0),
		HealthCheck: req.HealthCheck,
		CreatedAt:   time.Now(),
	}

	api.registry.Register(service)

	// Deploy containers
	for i := 0; i < req.Replicas; i++ {
		containerName := fmt.Sprintf("%s-%d", req.Name, i)

		// Select node
		node, err := api.scheduler.SelectNode(req)
		if err != nil {
			http.Error(w, fmt.Sprintf("Scheduling failed: %v", err), http.StatusInternalServerError)
			return
		}

		// Create and start container
		containerID, err := api.dockerClient.CreateContainer(r.Context(), req, containerName)
		if err != nil {
			http.Error(w, fmt.Sprintf("Container creation failed: %v", err), http.StatusInternalServerError)
			return
		}

		if err := api.dockerClient.StartContainer(r.Context(), containerID); err != nil {
			http.Error(w, fmt.Sprintf("Container start failed: %v", err), http.StatusInternalServerError)
			return
		}

		// Add to service
		container := models.Container{
			ID:          containerID,
			Name:        containerName,
			ServiceName: req.Name,
			NodeID:      node.ID,
			Image:       req.Image,
			Status:      "running",
			Health:      "unknown",
			CreatedAt:   time.Now(),
			StartedAt:   time.Now(),
		}

		api.registry.AddContainer(req.Name, container)

		// Start health monitoring if configured
		if req.HealthCheck.Type != "" {
			api.healthChecker.StartMonitoring(containerID, req.HealthCheck, api.handleContainerFailure)
		}
	}

	// Log event
	api.logEvent(models.Event{
		Timestamp: time.Now(),
		Type:      "deployment",
		Service:   req.Name,
		Message:   fmt.Sprintf("Deployed %d replicas", req.Replicas),
	})

	w.WriteHeader(http.StatusCreated)
	json.NewEncoder(w).Encode(service)
}

// ListServicesHandler returns all services
func (api *API) ListServicesHandler(w http.ResponseWriter, r *http.Request) {
	services := api.registry.List()
	json.NewEncoder(w).Encode(services)
}

// ScaleHandler scales a service
func (api *API) ScaleHandler(w http.ResponseWriter, r *http.Request) {
	vars := mux.Vars(r)
	serviceName := vars["name"]

	var req struct {
		Replicas int `json:"replicas"`
	}
	if err := json.NewDecoder(r.Body).Decode(&req); err != nil {
		http.Error(w, "Invalid request body", http.StatusBadRequest)
		return
	}

	service, err := api.registry.Get(serviceName)
	if err != nil {
		http.Error(w, "Service not found", http.StatusNotFound)
		return
	}

	currentReplicas := len(service.Containers)
	desiredReplicas := req.Replicas

	if desiredReplicas > currentReplicas {
		// Scale up
		// Implementation similar to deploy
		w.WriteHeader(http.StatusAccepted)
		json.NewEncoder(w).Encode(map[string]string{"message": "Scaling up..."})
	} else if desiredReplicas < currentReplicas {
		// Scale down
		w.WriteHeader(http.StatusAccepted)
		json.NewEncoder(w).Encode(map[string]string{"message": "Scaling down..."})
	} else {
		w.WriteHeader(http.StatusOK)
		json.NewEncoder(w).Encode(map[string]string{"message": "Already at desired scale"})
	}
}

// LogsHandler retrieves container logs
func (api *API) LogsHandler(w http.ResponseWriter, r *http.Request) {
	vars := mux.Vars(r)
	serviceName := vars["name"]

	service, err := api.registry.Get(serviceName)
	if err != nil {
		http.Error(w, "Service not found", http.StatusNotFound)
		return
	}

	if len(service.Containers) == 0 {
		http.Error(w, "No containers found", http.StatusNotFound)
		return
	}

	// Get logs from first container
	logs, err := api.dockerClient.GetContainerLogs(r.Context(), service.Containers[0].ID)
	if err != nil {
		http.Error(w, fmt.Sprintf("Failed to get logs: %v", err), http.StatusInternalServerError)
		return
	}
	defer logs.Close()

	w.Header().Set("Content-Type", "text/plain")
	io.Copy(w, logs)
}

// EventsHandler returns recent events
func (api *API) EventsHandler(w http.ResponseWriter, r *http.Request) {
	json.NewEncoder(w).Encode(api.eventLog)
}

// handleContainerFailure handles failed health checks
func (api *API) handleContainerFailure(containerID string) {
	// Find service and container
	services := api.registry.List()
	for _, service := range services {
		for _, container := range service.Containers {
			if container.ID == containerID {
				// Log event
				api.logEvent(models.Event{
					Timestamp: time.Now(),
					Type:      "health_check",
					Service:   service.Name,
					Container: containerID,
					Message:   "Container failed health check, restarting...",
				})

				// Restart container
				ctx := context.Background()
				api.dockerClient.StopContainer(ctx, containerID, 10*time.Second)
				api.dockerClient.StartContainer(ctx, containerID)

				return
			}
		}
	}
}

// logEvent adds an event to the event log
func (api *API) logEvent(event models.Event) {
	api.eventLog = append(api.eventLog, event)

	// Keep only last 100 events
	if len(api.eventLog) > 100 {
		api.eventLog = api.eventLog[len(api.eventLog)-100:]
	}
}
