package docker

import (
	"context"
	"fmt"
	"io"
	"time"

	"github.com/docker/docker/api/types"
	"github.com/docker/docker/api/types/container"
	"github.com/docker/docker/api/types/filters"
	"github.com/docker/docker/api/types/image"
	"github.com/docker/docker/api/types/network"
	"github.com/docker/docker/client"
	"github.com/docker/go-connections/nat"
	"github.com/yourusername/container-orchestrator/internal/models"
)

// Client wraps Docker SDK operations
type Client struct {
	cli *client.Client
}

// NewClient creates a new Docker client
func NewClient() (*Client, error) {
	cli, err := client.NewClientWithOpts(
		client.FromEnv,
		client.WithAPIVersionNegotiation(),
	)
	if err != nil {
		return nil, fmt.Errorf("failed to create docker client: %w", err)
	}

	return &Client{cli: cli}, nil
}

// CreateContainer creates a new container from a deployment request
func (c *Client) CreateContainer(ctx context.Context, req models.DeploymentRequest, name string) (string, error) {
	// Pull image if not exists
	if err := c.ensureImage(ctx, req.Image); err != nil {
		return "", fmt.Errorf("failed to ensure image: %w", err)
	}

	// Convert environment variables
	env := make([]string, 0, len(req.Env))
	for k, v := range req.Env {
		env = append(env, fmt.Sprintf("%s=%s", k, v))
	}

	// Configure container
	config := &container.Config{
		Image: req.Image,
		Cmd:   req.Command,
		Env:   env,
		Labels: map[string]string{
			"orchestrator.service": req.Name,
			"orchestrator.managed": "true",
		},
	}

	// Configure host settings (resource limits, port bindings)
	hostConfig := &container.HostConfig{
		Resources: container.Resources{
			CPUShares: req.Resources.CPUShares,
			Memory:    req.Resources.MemoryMB * 1024 * 1024, // Convert to bytes
		},
		RestartPolicy: container.RestartPolicy{
			Name:              "on-failure",
			MaximumRetryCount: 3,
		},
	}

	// Configure port bindings
	if len(req.Ports) > 0 {
		hostConfig.PortBindings = c.buildPortBindings(req.Ports)
	}

	// Create container
	resp, err := c.cli.ContainerCreate(
		ctx,
		config,
		hostConfig,
		&network.NetworkingConfig{},
		nil,
		name,
	)

	if err != nil {
		return "", fmt.Errorf("failed to create container: %w", err)
	}

	return resp.ID, nil
}

// StartContainer starts a created container
func (c *Client) StartContainer(ctx context.Context, containerID string) error {
	if err := c.cli.ContainerStart(ctx, containerID, container.StartOptions{}); err != nil {
		return fmt.Errorf("failed to start container: %w", err)
	}
	return nil
}

// StopContainer stops a running container
func (c *Client) StopContainer(ctx context.Context, containerID string, timeout time.Duration) error {
	timeoutSeconds := int(timeout.Seconds())
	if err := c.cli.ContainerStop(ctx, containerID, container.StopOptions{Timeout: &timeoutSeconds}); err != nil {
		return fmt.Errorf("failed to stop container: %w", err)
	}
	return nil
}

// RemoveContainer removes a container
func (c *Client) RemoveContainer(ctx context.Context, containerID string) error {
	if err := c.cli.ContainerRemove(ctx, containerID, container.RemoveOptions{
		Force: true,
	}); err != nil {
		return fmt.Errorf("failed to remove container: %w", err)
	}
	return nil
}

// GetContainerLogs retrieves container logs
func (c *Client) GetContainerLogs(ctx context.Context, containerID string) (io.ReadCloser, error) {
	return c.cli.ContainerLogs(ctx, containerID, container.LogsOptions{
		ShowStdout: true,
		ShowStderr: true,
		Timestamps: true,
		Tail:       "100",
	})
}

// InspectContainer gets detailed container information
func (c *Client) InspectContainer(ctx context.Context, containerID string) (*types.ContainerJSON, error) {
	inspect, err := c.cli.ContainerInspect(ctx, containerID)
	if err != nil {
		return nil, fmt.Errorf("failed to inspect container: %w", err)
	}
	return &inspect, nil
}

// ListContainers lists all containers managed by orchestrator
func (c *Client) ListContainers(ctx context.Context) ([]types.Container, error) {
	containers, err := c.cli.ContainerList(ctx, container.ListOptions{
		All:     true,
		Filters: filters.NewArgs(filters.Arg("label", "orchestrator.managed=true")),
	})
	if err != nil {
		return nil, fmt.Errorf("failed to list containers: %w", err)
	}
	return containers, nil
}

// ensureImage pulls image if not present locally
func (c *Client) ensureImage(ctx context.Context, imageName string) error {
	images, err := c.cli.ImageList(ctx, image.ListOptions{})
	if err != nil {
		return err
	}

	// Check if image exists
	for _, img := range images {
		for _, tag := range img.RepoTags {
			if tag == imageName {
				return nil // Image already exists
			}
		}
	}

	// Pull image
	reader, err := c.cli.ImagePull(ctx, imageName, image.PullOptions{})
	if err != nil {
		return err
	}
	defer reader.Close()

	// Consume pull output (required to complete pull)
	io.Copy(io.Discard, reader)
	return nil
}

// buildPortBindings converts port mappings to Docker format
func (c *Client) buildPortBindings(ports []models.PortMapping) nat.PortMap {
	portBindings := nat.PortMap{}
	for _, p := range ports {
		port := nat.Port(fmt.Sprintf("%d/%s", p.ContainerPort, p.Protocol))
		portBindings[port] = []nat.PortBinding{
			{
				HostIP:   "0.0.0.0",
				HostPort: fmt.Sprintf("%d", p.HostPort),
			},
		}
	}
	return portBindings
}

// Close closes the Docker client
func (c *Client) Close() error {
	return c.cli.Close()
}
