package health

import (
	"context"
	"net"
	"net/http"
	"sync"
	"time"

	"github.com/yourusername/container-orchestrator/internal/models"
)

// Checker monitors container health
type Checker struct {
	checks map[string]*HealthMonitor
	mu     sync.RWMutex
}

// HealthMonitor tracks health for a single container
type HealthMonitor struct {
	ContainerID  string
	Config       models.HealthCheckConfig
	Status       string
	FailureCount int
	LastCheck    time.Time
	StopCh       chan struct{}
}

// NewChecker creates a new health checker
func NewChecker() *Checker {
	return &Checker{
		checks: make(map[string]*HealthMonitor),
	}
}

// StartMonitoring begins health checking for a container
func (c *Checker) StartMonitoring(containerID string, config models.HealthCheckConfig, onFailure func(string)) {
	c.mu.Lock()
	defer c.mu.Unlock()

	// Stop existing monitor if any
	if existing, exists := c.checks[containerID]; exists {
		close(existing.StopCh)
	}

	monitor := &HealthMonitor{
		ContainerID: containerID,
		Config:      config,
		Status:      "unknown",
		StopCh:      make(chan struct{}),
	}

	c.checks[containerID] = monitor

	// Start monitoring goroutine
	go c.runHealthChecks(monitor, onFailure)
}

// StopMonitoring stops health checking for a container
func (c *Checker) StopMonitoring(containerID string) {
	c.mu.Lock()
	defer c.mu.Unlock()

	if monitor, exists := c.checks[containerID]; exists {
		close(monitor.StopCh)
		delete(c.checks, containerID)
	}
}

// GetStatus returns the current health status
func (c *Checker) GetStatus(containerID string) string {
	c.mu.RLock()
	defer c.mu.RUnlock()

	if monitor, exists := c.checks[containerID]; exists {
		return monitor.Status
	}
	return "unknown"
}

// runHealthChecks performs periodic health checks
func (c *Checker) runHealthChecks(monitor *HealthMonitor, onFailure func(string)) {
	ticker := time.NewTicker(monitor.Config.Interval)
	defer ticker.Stop()

	for {
		select {
		case <-monitor.StopCh:
			return
		case <-ticker.C:
			healthy := c.performCheck(monitor)

			c.mu.Lock()
			monitor.LastCheck = time.Now()

			if healthy {
				monitor.Status = "healthy"
				monitor.FailureCount = 0
			} else {
				monitor.FailureCount++
				if monitor.FailureCount >= monitor.Config.Retries {
					monitor.Status = "unhealthy"
					c.mu.Unlock()

					// Trigger failure callback
					onFailure(monitor.ContainerID)
					continue
				}
			}
			c.mu.Unlock()
		}
	}
}

// performCheck executes a single health check
func (c *Checker) performCheck(monitor *HealthMonitor) bool {
	ctx, cancel := context.WithTimeout(context.Background(), monitor.Config.Timeout)
	defer cancel()

	switch monitor.Config.Type {
	case "http":
		return c.checkHTTP(ctx, monitor.Config.Endpoint)
	case "tcp":
		return c.checkTCP(ctx, monitor.Config.Endpoint)
	case "exec":
		return c.checkExec(ctx, monitor.ContainerID, monitor.Config.Command)
	default:
		return false
	}
}

// checkHTTP performs HTTP health check
func (c *Checker) checkHTTP(ctx context.Context, endpoint string) bool {
	req, err := http.NewRequestWithContext(ctx, "GET", endpoint, nil)
	if err != nil {
		return false
	}

	resp, err := http.DefaultClient.Do(req)
	if err != nil {
		return false
	}
	defer resp.Body.Close()

	return resp.StatusCode >= 200 && resp.StatusCode < 300
}

// checkTCP performs TCP connection check
func (c *Checker) checkTCP(ctx context.Context, address string) bool {
	var d net.Dialer
	conn, err := d.DialContext(ctx, "tcp", address)
	if err != nil {
		return false
	}
	conn.Close()
	return true
}

// checkExec performs command execution check (simplified)
func (c *Checker) checkExec(ctx context.Context, containerID string, command []string) bool {
	// In a full implementation, this would use Docker exec API
	// For simplicity, we'll return true
	return true
}
