package models

import (
	"time"
)

// DeploymentRequest represents a service deployment request
type DeploymentRequest struct {
	Name        string            `json:"name"`
	Image       string            `json:"image"`
	Replicas    int               `json:"replicas"`
	Command     []string          `json:"command,omitempty"`
	Env         map[string]string `json:"env,omitempty"`
	Ports       []PortMapping     `json:"ports,omitempty"`
	Resources   ResourceLimits    `json:"resources,omitempty"`
	HealthCheck HealthCheckConfig `json:"health_check,omitempty"`
}

// PortMapping defines container port exposure
type PortMapping struct {
	ContainerPort int    `json:"container_port"`
	HostPort      int    `json:"host_port"`
	Protocol      string `json:"protocol"` // tcp, udp
}

// ResourceLimits defines CPU and memory constraints
type ResourceLimits struct {
	CPUShares    int64 `json:"cpu_shares"`              // CPU shares (relative weight)
	MemoryMB     int64 `json:"memory_mb"`               // Memory limit in MB
	MemorySwapMB int64 `json:"memory_swap_mb,omitempty"`
}

// HealthCheckConfig defines health monitoring
type HealthCheckConfig struct {
	Type     string        `json:"type"`              // http, tcp, exec
	Endpoint string        `json:"endpoint"`          // URL for http, port for tcp
	Command  []string      `json:"command,omitempty"` // Command for exec
	Interval time.Duration `json:"interval"`
	Timeout  time.Duration `json:"timeout"`
	Retries  int           `json:"retries"`
}

// Container represents a running container instance
type Container struct {
	ID          string    `json:"id"`
	Name        string    `json:"name"`
	ServiceName string    `json:"service_name"`
	NodeID      string    `json:"node_id"`
	Image       string    `json:"image"`
	Status      string    `json:"status"` // running, stopped, failed
	Health      string    `json:"health"` // healthy, unhealthy, unknown
	CreatedAt   time.Time `json:"created_at"`
	StartedAt   time.Time `json:"started_at"`
}

// Service represents a deployed service with multiple replicas
type Service struct {
	Name        string            `json:"name"`
	Image       string            `json:"image"`
	Replicas    int               `json:"replicas"`
	Containers  []Container       `json:"containers"`
	HealthCheck HealthCheckConfig `json:"health_check"`
	CreatedAt   time.Time         `json:"created_at"`
}

// Node represents a host machine that can run containers
type Node struct {
	ID        string    `json:"id"`
	Address   string    `json:"address"`
	CPUCores  int       `json:"cpu_cores"`
	MemoryMB  int64     `json:"memory_mb"`
	Available bool      `json:"available"`
	LastSeen  time.Time `json:"last_seen"`
}

// Placement represents a container scheduled to a node
type Placement struct {
	ContainerID string `json:"container_id"`
	NodeID      string `json:"node_id"`
	ServiceName string `json:"service_name"`
}

// Event represents a system event
type Event struct {
	Timestamp time.Time `json:"timestamp"`
	Type      string    `json:"type"` // deployment, health_check, restart
	Service   string    `json:"service"`
	Container string    `json:"container,omitempty"`
	Message   string    `json:"message"`
}
