package registry

import (
	"errors"
	"sync"

	"github.com/yourusername/container-orchestrator/internal/models"
)

var (
	ErrServiceNotFound = errors.New("service not found")
)

// Registry manages service definitions and discovery
type Registry struct {
	services map[string]*models.Service
	mu       sync.RWMutex
}

// NewRegistry creates a new service registry
func NewRegistry() *Registry {
	return &Registry{
		services: make(map[string]*models.Service),
	}
}

// Register adds or updates a service
func (r *Registry) Register(service *models.Service) {
	r.mu.Lock()
	defer r.mu.Unlock()
	r.services[service.Name] = service
}

// Unregister removes a service
func (r *Registry) Unregister(serviceName string) {
	r.mu.Lock()
	defer r.mu.Unlock()
	delete(r.services, serviceName)
}

// Get retrieves a service by name
func (r *Registry) Get(serviceName string) (*models.Service, error) {
	r.mu.RLock()
	defer r.mu.RUnlock()

	service, exists := r.services[serviceName]
	if !exists {
		return nil, ErrServiceNotFound
	}
	return service, nil
}

// List returns all services
func (r *Registry) List() []*models.Service {
	r.mu.RLock()
	defer r.mu.RUnlock()

	services := make([]*models.Service, 0, len(r.services))
	for _, service := range r.services {
		services = append(services, service)
	}
	return services
}

// AddContainer adds a container to a service
func (r *Registry) AddContainer(serviceName string, container models.Container) error {
	r.mu.Lock()
	defer r.mu.Unlock()

	service, exists := r.services[serviceName]
	if !exists {
		return ErrServiceNotFound
	}

	service.Containers = append(service.Containers, container)
	return nil
}

// RemoveContainer removes a container from a service
func (r *Registry) RemoveContainer(serviceName, containerID string) error {
	r.mu.Lock()
	defer r.mu.Unlock()

	service, exists := r.services[serviceName]
	if !exists {
		return ErrServiceNotFound
	}

	for i, container := range service.Containers {
		if container.ID == containerID {
			service.Containers = append(service.Containers[:i], service.Containers[i+1:]...)
			return nil
		}
	}

	return nil
}

// Discover returns all container IPs for a service
func (r *Registry) Discover(serviceName string) ([]string, error) {
	r.mu.RLock()
	defer r.mu.RUnlock()

	service, exists := r.services[serviceName]
	if !exists {
		return nil, ErrServiceNotFound
	}

	endpoints := make([]string, 0, len(service.Containers))
	for _, container := range service.Containers {
		if container.Status == "running" && container.Health == "healthy" {
			endpoints = append(endpoints, container.ID) // In production, return IP:Port
		}
	}

	return endpoints, nil
}
