package scheduler

import (
	"errors"
	"fmt"
	"sync"

	"github.com/yourusername/container-orchestrator/internal/models"
)

var (
	ErrNoSuitableNode = errors.New("no suitable node found")
)

// Scheduler handles container placement across nodes
type Scheduler struct {
	nodes map[string]*models.Node
	mu    sync.RWMutex
}

// NewScheduler creates a new scheduler
func NewScheduler() *Scheduler {
	return &Scheduler{
		nodes: make(map[string]*models.Node),
	}
}

// RegisterNode adds a node to the scheduler
func (s *Scheduler) RegisterNode(node *models.Node) {
	s.mu.Lock()
	defer s.mu.Unlock()
	s.nodes[node.ID] = node
}

// UnregisterNode removes a node from the scheduler
func (s *Scheduler) UnregisterNode(nodeID string) {
	s.mu.Lock()
	defer s.mu.Unlock()
	delete(s.nodes, nodeID)
}

// SelectNode chooses the best node for a container
func (s *Scheduler) SelectNode(req models.DeploymentRequest) (*models.Node, error) {
	s.mu.RLock()
	defer s.mu.RUnlock()

	cpuReq := req.Resources.CPUShares
	if cpuReq == 0 {
		cpuReq = 1024 // Default CPU shares
	}

	memReq := req.Resources.MemoryMB
	if memReq == 0 {
		memReq = 512 // Default 512MB
	}

	// Find node with sufficient resources
	var bestNode *models.Node
	var maxAvailableMemory int64

	for _, node := range s.nodes {
		if !node.Available {
			continue
		}

		// Check if node has sufficient resources
		if node.MemoryMB >= memReq {
			// Select node with most available memory (simple strategy)
			if bestNode == nil || node.MemoryMB > maxAvailableMemory {
				bestNode = node
				maxAvailableMemory = node.MemoryMB
			}
		}
	}

	if bestNode == nil {
		return nil, ErrNoSuitableNode
	}

	return bestNode, nil
}

// GetNodes returns all registered nodes
func (s *Scheduler) GetNodes() []*models.Node {
	s.mu.RLock()
	defer s.mu.RUnlock()

	nodes := make([]*models.Node, 0, len(s.nodes))
	for _, node := range s.nodes {
		nodes = append(nodes, node)
	}
	return nodes
}

// GetNode returns a specific node
func (s *Scheduler) GetNode(nodeID string) (*models.Node, error) {
	s.mu.RLock()
	defer s.mu.RUnlock()

	node, exists := s.nodes[nodeID]
	if !exists {
		return nil, fmt.Errorf("node %s not found", nodeID)
	}
	return node, nil
}
