package tests

import (
	"bytes"
	"context"
	"encoding/json"
	"net/http"
	"net/http/httptest"
	"testing"
	"time"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"github.com/yourusername/container-orchestrator/internal/api"
	"github.com/yourusername/container-orchestrator/internal/docker"
	"github.com/yourusername/container-orchestrator/internal/health"
	"github.com/yourusername/container-orchestrator/internal/models"
	"github.com/yourusername/container-orchestrator/internal/registry"
	"github.com/yourusername/container-orchestrator/internal/scheduler"
)

func TestDeployService(t *testing.T) {
	// Setup
	dockerClient, err := docker.NewClient()
	require.NoError(t, err)
	defer dockerClient.Close()

	sched := scheduler.NewScheduler()
	reg := registry.NewRegistry()
	checker := health.NewChecker()

	// Register test node
	sched.RegisterNode(&models.Node{
		ID:        "test-node",
		Address:   "localhost",
		CPUCores:  2,
		MemoryMB:  2048,
		Available: true,
	})

	apiHandler := api.NewAPI(dockerClient, sched, reg, checker)

	// Prepare deployment request
	deployReq := models.DeploymentRequest{
		Name:     "test-service",
		Image:    "nginx:alpine",
		Replicas: 2,
	}

	body, _ := json.Marshal(deployReq)
	req := httptest.NewRequest("POST", "/api/deploy", bytes.NewReader(body))
	w := httptest.NewRecorder()

	// Execute
	apiHandler.DeployHandler(w, req)

	// Assert
	assert.Equal(t, http.StatusCreated, w.Code)

	// Verify service registered
	service, err := reg.Get("test-service")
	require.NoError(t, err)
	assert.Equal(t, "test-service", service.Name)
	assert.Equal(t, 2, len(service.Containers))

	// Cleanup
	ctx := context.Background()
	for _, container := range service.Containers {
		dockerClient.StopContainer(ctx, container.ID, 5*time.Second)
		dockerClient.RemoveContainer(ctx, container.ID)
	}
}

func TestSchedulerNodeSelection(t *testing.T) {
	sched := scheduler.NewScheduler()

	// Register nodes
	sched.RegisterNode(&models.Node{
		ID:        "node1",
		CPUCores:  2,
		MemoryMB:  1024,
		Available: true,
	})

	sched.RegisterNode(&models.Node{
		ID:        "node2",
		CPUCores:  4,
		MemoryMB:  4096,
		Available: true,
	})

	// Request with high memory
	req := models.DeploymentRequest{
		Resources: models.ResourceLimits{
			MemoryMB: 2048,
		},
	}

	node, err := sched.SelectNode(req)
	require.NoError(t, err)
	assert.Equal(t, "node2", node.ID) // Should select node with more memory
}

func TestHealthChecking(t *testing.T) {
	checker := health.NewChecker()

	failureCount := 0
	onFailure := func(containerID string) {
		failureCount++
	}

	// Start monitoring with HTTP check
	checker.StartMonitoring("test-container", models.HealthCheckConfig{
		Type:     "http",
		Endpoint: "http://localhost:9999/health", // Non-existent endpoint
		Interval: 1 * time.Second,
		Timeout:  500 * time.Millisecond,
		Retries:  2,
	}, onFailure)

	// Wait for health checks to fail
	time.Sleep(3 * time.Second)

	// Assert failure was detected
	assert.Greater(t, failureCount, 0)
	assert.Equal(t, "unhealthy", checker.GetStatus("test-container"))

	// Cleanup
	checker.StopMonitoring("test-container")
}
