package main

import (
	"flag"
	"fmt"
	"log"
	"os"
	"os/signal"
	"strings"
	"syscall"

	"github.com/distributed-db/pkg/raft"
	"github.com/distributed-db/pkg/storage"
	"github.com/distributed-db/pkg/transaction"
)

func main() {
	// Command-line flags
	nodeID := flag.Int("node-id", 1, "Node ID (1-based)")
	listenAddr := flag.String("listen-addr", ":9001", "Listen address")
	dataDir := flag.String("data-dir", "/tmp/data/node1", "Data directory")
	peers := flag.String("peers", "", "Comma-separated peer addresses (e.g., node2:9002,node3:9003)")
	flag.Parse()

	log.Printf("Starting database node %d on %s", *nodeID, *listenAddr)

	// Parse peers
	var peerAddrs []string
	if *peers != "" {
		peerAddrs = strings.Split(*peers, ",")
	}

	// Create data directory
	if err := os.MkdirAll(*dataDir, 0755); err != nil {
		log.Fatalf("Failed to create data directory: %v", err)
	}

	// Initialize LSM-tree storage engine
	lsmConfig := storage.DefaultLSMConfig()
	lsmConfig.DataDir = *dataDir
	lsm, err := storage.NewLSMTree(lsmConfig)
	if err != nil {
		log.Fatalf("Failed to create LSM-tree: %v", err)
	}
	defer lsm.Close()

	log.Printf("LSM-tree initialized at %s", *dataDir)

	// Initialize Raft consensus
	raftConfig := raft.DefaultConfig()
	raftConfig.NodeID = *nodeID
	raftConfig.ListenAddr = *listenAddr
	raftConfig.Peers = peerAddrs
	raftNode, err := raft.NewRaftNode(raftConfig, lsm)
	if err != nil {
		log.Fatalf("Failed to create Raft node: %v", err)
	}
	defer raftNode.Stop()

	log.Printf("Raft node initialized with peers: %v", peerAddrs)

	// Initialize transaction coordinator
	txnCoord := transaction.NewCoordinator(raftNode, lsm)
	defer txnCoord.Stop()

	log.Printf("Transaction coordinator initialized")

	// Start background tasks
	go lsm.RunCompaction()
	go raftNode.Run()
	go txnCoord.Run()

	log.Printf("Database node %d is ready!", *nodeID)

	// Wait for shutdown signal
	sigChan := make(chan os.Signal, 1)
	signal.Notify(sigChan, syscall.SIGINT, syscall.SIGTERM)
	<-sigChan

	log.Println("Shutting down gracefully...")
	fmt.Println("Shutdown complete")
}
