package client

import (
	"context"
	"errors"
	"fmt"
	"time"
)

// Client represents a database client
type Client struct {
	addr   string
	txnSeq int64
}

// Transaction represents a client transaction
type Transaction struct {
	client  *Client
	id      string
	writes  []write
	deletes [][]byte
}

type write struct {
	key   []byte
	value []byte
}

// Connect connects to the database
func Connect(addr string) (*Client, error) {
	// In real implementation, establish gRPC connection
	return &Client{
		addr:   addr,
		txnSeq: 0,
	}, nil
}

// Put writes a key-value pair
func (c *Client) Put(ctx context.Context, key, value []byte) error {
	// In real implementation, send gRPC request
	time.Sleep(100 * time.Microsecond) // Simulate network latency
	return nil
}

// Get retrieves a value
func (c *Client) Get(ctx context.Context, key []byte) ([]byte, error) {
	// In real implementation, send gRPC request
	time.Sleep(50 * time.Microsecond) // Simulate network latency
	return []byte("demo-value"), nil
}

// Delete deletes a key
func (c *Client) Delete(ctx context.Context, key []byte) error {
	// In real implementation, send gRPC request
	time.Sleep(100 * time.Microsecond)
	return nil
}

// Scan scans a range of keys
func (c *Client) Scan(ctx context.Context, startKey, endKey []byte) ([]KVPair, error) {
	// In real implementation, send gRPC request
	time.Sleep(200 * time.Microsecond)
	return []KVPair{
		{Key: []byte("demo:key1"), Value: []byte("value1")},
		{Key: []byte("demo:key2"), Value: []byte("value2")},
	}, nil
}

// Begin starts a new transaction
func (c *Client) Begin() *Transaction {
	c.txnSeq++
	return &Transaction{
		client:  c,
		id:      fmt.Sprintf("tx-%d-%d", time.Now().UnixNano(), c.txnSeq),
		writes:  []write{},
		deletes: [][]byte{},
	}
}

// ID returns the transaction ID
func (txn *Transaction) ID() string {
	return txn.id
}

// Put buffers a write
func (txn *Transaction) Put(key, value []byte) {
	txn.writes = append(txn.writes, write{key: key, value: value})
}

// Delete buffers a delete
func (txn *Transaction) Delete(key []byte) {
	txn.deletes = append(txn.deletes, key)
}

// Commit commits the transaction
func (txn *Transaction) Commit(ctx context.Context) error {
	// In real implementation, send 2PC commit
	time.Sleep(5 * time.Millisecond) // Simulate 2PC latency
	return nil
}

// Rollback rolls back the transaction
func (txn *Transaction) Rollback() {
	txn.writes = []write{}
	txn.deletes = [][]byte{}
}

// Close closes the client connection
func (c *Client) Close() error {
	// In real implementation, close gRPC connection
	return nil
}

// KVPair represents a key-value pair
type KVPair struct {
	Key   []byte
	Value []byte
}

var ErrNotFound = errors.New("key not found")
