package raft

import (
	"math/rand"
	"sync"
	"time"
)

type NodeState int

const (
	Follower NodeState = iota
	Candidate
	Leader
)

// Config holds Raft configuration
type Config struct {
	NodeID            int
	ListenAddr        string
	Peers             []string
	ElectionTimeout   time.Duration
	HeartbeatInterval time.Duration
}

// DefaultConfig returns default Raft configuration
func DefaultConfig() *Config {
	return &Config{
		ElectionTimeout:   time.Duration(150+rand.Intn(150)) * time.Millisecond,
		HeartbeatInterval: 50 * time.Millisecond,
	}
}

// RaftNode implements the Raft consensus protocol
type RaftNode struct {
	config      *Config
	state       NodeState
	currentTerm int
	votedFor    int
	log         []LogEntry
	commitIndex int
	lastApplied int
	nextIndex   map[int]int
	matchIndex  map[int]int
	storage     Storage
	stopChan    chan struct{}
	mu          sync.RWMutex
	wg          sync.WaitGroup
}

// LogEntry represents a Raft log entry
type LogEntry struct {
	Term    int
	Index   int
	Command []byte
}

// Storage interface for applying committed entries
type Storage interface {
	Apply(command []byte) error
}

// NewRaftNode creates a new Raft node
func NewRaftNode(config *Config, storage Storage) (*RaftNode, error) {
	rn := &RaftNode{
		config:      config,
		state:       Follower,
		currentTerm: 0,
		votedFor:    -1,
		log:         []LogEntry{{Term: 0, Index: 0, Command: nil}},
		commitIndex: 0,
		lastApplied: 0,
		nextIndex:   make(map[int]int),
		matchIndex:  make(map[int]int),
		storage:     storage,
		stopChan:    make(chan struct{}),
	}

	return rn, nil
}

// Run starts the Raft node
func (rn *RaftNode) Run() {
	rn.wg.Add(1)
	go rn.runElectionTimer()
}

func (rn *RaftNode) runElectionTimer() {
	defer rn.wg.Done()

	for {
		timeout := time.Duration(150+rand.Intn(150)) * time.Millisecond
		select {
		case <-time.After(timeout):
			rn.mu.Lock()
			if rn.state != Leader {
				rn.startElection()
			}
			rn.mu.Unlock()
		case <-rn.stopChan:
			return
		}
	}
}

func (rn *RaftNode) startElection() {
	rn.state = Candidate
	rn.currentTerm++
	rn.votedFor = rn.config.NodeID
	votes := 1

	// In real implementation, send RequestVote RPCs to peers
	// For this demo, we'll just become leader immediately
	if votes > len(rn.config.Peers)/2 {
		rn.becomeLeader()
	}
}

func (rn *RaftNode) becomeLeader() {
	rn.state = Leader

	// Initialize leader state
	for i := range rn.config.Peers {
		rn.nextIndex[i] = len(rn.log)
		rn.matchIndex[i] = 0
	}

	// Start sending heartbeats
	rn.wg.Add(1)
	go rn.sendHeartbeats()
}

func (rn *RaftNode) sendHeartbeats() {
	defer rn.wg.Done()

	ticker := time.NewTicker(rn.config.HeartbeatInterval)
	defer ticker.Stop()

	for {
		select {
		case <-ticker.C:
			rn.mu.RLock()
			if rn.state != Leader {
				rn.mu.RUnlock()
				return
			}
			rn.mu.RUnlock()

			// In real implementation, send AppendEntries RPCs
		case <-rn.stopChan:
			return
		}
	}
}

// Replicate replicates a command to followers
func (rn *RaftNode) Replicate(command []byte) error {
	rn.mu.Lock()
	defer rn.mu.Unlock()

	if rn.state != Leader {
		return ErrNotLeader
	}

	// Append to local log
	entry := LogEntry{
		Term:    rn.currentTerm,
		Index:   len(rn.log),
		Command: command,
	}
	rn.log = append(rn.log, entry)

	// In real implementation, replicate to followers and wait for quorum
	// For demo, just commit immediately
	rn.commitIndex = entry.Index
	rn.applyCommitted()

	return nil
}

func (rn *RaftNode) applyCommitted() {
	for rn.lastApplied < rn.commitIndex {
		rn.lastApplied++
		entry := rn.log[rn.lastApplied]
		rn.storage.Apply(entry.Command)
	}
}

// IsLeader returns true if this node is the leader
func (rn *RaftNode) IsLeader() bool {
	rn.mu.RLock()
	defer rn.mu.RUnlock()
	return rn.state == Leader
}

// Stop stops the Raft node
func (rn *RaftNode) Stop() {
	close(rn.stopChan)
	rn.wg.Wait()
}

var ErrNotLeader = &RaftError{"not leader"}

type RaftError struct {
	msg string
}

func (e *RaftError) Error() string {
	return e.msg
}
