package storage

import (
	"github.com/spaolacci/murmur3"
)

// BloomFilter is a probabilistic data structure for set membership
type BloomFilter struct {
	bits     []byte
	numBits  uint64
	numHash  int
}

// NewBloomFilter creates a new bloom filter
func NewBloomFilter(expectedKeys int, bitsPerKey int) *BloomFilter {
	numBits := uint64(expectedKeys * bitsPerKey)
	numHash := int(float64(bitsPerKey) * 0.69) // ln(2) ≈ 0.69

	if numHash < 1 {
		numHash = 1
	}
	if numHash > 30 {
		numHash = 30
	}

	return &BloomFilter{
		bits:    make([]byte, (numBits+7)/8),
		numBits: numBits,
		numHash: numHash,
	}
}

// Add adds a key to the bloom filter
func (bf *BloomFilter) Add(key []byte) {
	h1, h2 := bf.hash(key)
	for i := 0; i < bf.numHash; i++ {
		pos := (h1 + uint64(i)*h2) % bf.numBits
		bf.bits[pos/8] |= (1 << (pos % 8))
	}
}

// MayContain checks if a key might be in the set
func (bf *BloomFilter) MayContain(key []byte) bool {
	h1, h2 := bf.hash(key)
	for i := 0; i < bf.numHash; i++ {
		pos := (h1 + uint64(i)*h2) % bf.numBits
		if bf.bits[pos/8]&(1<<(pos%8)) == 0 {
			return false
		}
	}
	return true
}

func (bf *BloomFilter) hash(key []byte) (uint64, uint64) {
	h1, h2 := murmur3.Sum128(key)
	return h1, h2
}
