package storage

import (
	"container/list"
	"sync"
)

// BlockCache is an LRU cache for SSTable blocks
type BlockCache struct {
	maxSize int64
	size    int64
	items   map[string]*list.Element
	lru     *list.List
	mu      sync.RWMutex
}

type cacheEntry struct {
	key   string
	value []byte
	size  int64
}

// NewBlockCache creates a new block cache
func NewBlockCache(maxSize int64) *BlockCache {
	return &BlockCache{
		maxSize: maxSize,
		size:    0,
		items:   make(map[string]*list.Element),
		lru:     list.New(),
	}
}

// Get retrieves a value from cache
func (bc *BlockCache) Get(fileID uint64, key []byte) ([]byte, bool) {
	bc.mu.RLock()
	defer bc.mu.RUnlock()

	cacheKey := cacheKey(fileID, key)
	elem, found := bc.items[cacheKey]
	if !found {
		return nil, false
	}

	bc.lru.MoveToFront(elem)
	entry := elem.Value.(*cacheEntry)
	return entry.value, true
}

// Put adds a value to cache
func (bc *BlockCache) Put(fileID uint64, key, value []byte) {
	bc.mu.Lock()
	defer bc.mu.Unlock()

	cacheKey := cacheKey(fileID, key)
	size := int64(len(key) + len(value))

	// Check if already cached
	if elem, found := bc.items[cacheKey]; found {
		bc.lru.MoveToFront(elem)
		entry := elem.Value.(*cacheEntry)
		entry.value = value
		return
	}

	// Evict if necessary
	for bc.size+size > bc.maxSize && bc.lru.Len() > 0 {
		bc.evictOldest()
	}

	// Add new entry
	entry := &cacheEntry{
		key:   cacheKey,
		value: value,
		size:  size,
	}
	elem := bc.lru.PushFront(entry)
	bc.items[cacheKey] = elem
	bc.size += size
}

func (bc *BlockCache) evictOldest() {
	elem := bc.lru.Back()
	if elem != nil {
		bc.lru.Remove(elem)
		entry := elem.Value.(*cacheEntry)
		delete(bc.items, entry.key)
		bc.size -= entry.size
	}
}

func cacheKey(fileID uint64, key []byte) string {
	return string(append([]byte{byte(fileID)}, key...))
}
