package storage

import (
	"bytes"
	"encoding/binary"
	"fmt"
	"os"

	"github.com/golang/snappy"
)

// SSTable represents a sorted string table on disk
type SSTable struct {
	fileID      uint64
	level       int
	path        string
	minKey      []byte
	maxKey      []byte
	bloomFilter *BloomFilter
	index       []IndexEntry
	file        *os.File
}

type IndexEntry struct {
	key    []byte
	offset int64
	size   int64
}

// NewSSTableWriter creates a new SSTable writer
func NewSSTableWriter(path string, fileID uint64, level int, bloomBits int) (*SSTableWriter, error) {
	file, err := os.Create(path)
	if err != nil {
		return nil, err
	}

	return &SSTableWriter{
		path:        path,
		fileID:      fileID,
		level:       level,
		file:        file,
		bloomFilter: NewBloomFilter(10000, bloomBits),
		index:       []IndexEntry{},
	}, nil
}

type SSTableWriter struct {
	path        string
	fileID      uint64
	level       int
	file        *os.File
	bloomFilter *BloomFilter
	index       []IndexEntry
	minKey      []byte
	maxKey      []byte
	offset      int64
}

// Add adds a key-value pair
func (w *SSTableWriter) Add(key, value []byte) error {
	if w.minKey == nil {
		w.minKey = key
	}
	w.maxKey = key

	w.bloomFilter.Add(key)

	// Compress value
	compressed := snappy.Encode(nil, value)

	// Write: [keyLen:4][key][valueLen:4][compressedValue]
	buf := make([]byte, 4+len(key)+4+len(compressed))
	offset := 0

	binary.BigEndian.PutUint32(buf[offset:], uint32(len(key)))
	offset += 4
	copy(buf[offset:], key)
	offset += len(key)

	binary.BigEndian.PutUint32(buf[offset:], uint32(len(compressed)))
	offset += 4
	copy(buf[offset:], compressed)

	n, err := w.file.Write(buf)
	if err != nil {
		return err
	}

	w.index = append(w.index, IndexEntry{
		key:    key,
		offset: w.offset,
		size:   int64(n),
	})

	w.offset += int64(n)
	return nil
}

// Finalize closes the SSTable and returns reader
func (w *SSTableWriter) Finalize() (*SSTable, error) {
	w.file.Close()

	file, err := os.Open(w.path)
	if err != nil {
		return nil, err
	}

	return &SSTable{
		fileID:      w.fileID,
		level:       w.level,
		path:        w.path,
		minKey:      w.minKey,
		maxKey:      w.maxKey,
		bloomFilter: w.bloomFilter,
		index:       w.index,
		file:        file,
	}, nil
}

// Get retrieves a value by key
func (sst *SSTable) Get(key []byte) ([]byte, bool) {
	// Binary search in index
	left, right := 0, len(sst.index)-1
	var entry *IndexEntry

	for left <= right {
		mid := (left + right) / 2
		cmp := bytes.Compare(key, sst.index[mid].key)
		if cmp == 0 {
			entry = &sst.index[mid]
			break
		} else if cmp < 0 {
			right = mid - 1
		} else {
			left = mid + 1
		}
	}

	if entry == nil {
		return nil, false
	}

	// Read from file
	buf := make([]byte, entry.size)
	if _, err := sst.file.ReadAt(buf, entry.offset); err != nil {
		return nil, false
	}

	// Parse entry
	keyLen := binary.BigEndian.Uint32(buf[0:4])
	valueLen := binary.BigEndian.Uint32(buf[4+keyLen : 4+keyLen+4])
	compressedValue := buf[4+keyLen+4 : 4+keyLen+4+valueLen]

	// Decompress
	value, err := snappy.Decode(nil, compressedValue)
	if err != nil {
		return nil, false
	}

	return value, true
}

// Scan iterates over keys in range
func (sst *SSTable) Scan(startKey, endKey []byte, fn func(key, value []byte)) {
	for _, entry := range sst.index {
		if startKey != nil && bytes.Compare(entry.key, startKey) < 0 {
			continue
		}
		if endKey != nil && bytes.Compare(entry.key, endKey) >= 0 {
			break
		}

		value, found := sst.Get(entry.key)
		if found {
			fn(entry.key, value)
		}
	}
}

func formatSSTableFilename(fileID uint64, level int) string {
	return fmt.Sprintf("L%d-%016x.sst", level, fileID)
}
